<?php
namespace IssetBV\XAuthBundle\Service\Endpoint;

use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\XAuthBundle\Model\LinkInterface;
use IssetBV\XAuthBundle\Entity\Endpoint as EndpointEntity;
use IssetBV\XAuthBundle\Entity\EndpointToken;
use IssetBV\XAuthBundle\Repository\EndpointRepository;
use Doctrine\Bundle\DoctrineBundle\Registry;

/**
 * Class Endpoint.
 *
 * @author Felix Balfoort <felix@isset.nl>
 * @author Bart Malestein <bart@isset.nl>
 */
class Endpoint
{

    /**
     * @var string
     */
    private $salt;

    /**
     * @var EndpointRepository
     */
    private $endpointRepository;

    /**
     * @var ObjectManager
     */
    private $entityManager;

    /**
     *
     * @param Registry $doctrine
     * @param string $salt
     */
    public function __construct(Registry $doctrine, $salt)
    {
        $this->salt = $salt;
        $this->entityManager = $doctrine->getManager();
        $this->endpointRepository = $doctrine->getRepository("IssetBVXAuthBundle:Endpoint");
    }

    /**
     * @param LinkInterface $link
     * @param bool $autoFlush
     * @param string $hasher
     * @return EndpointEntity
     */
    public function findActiveForLink(LinkInterface $link, $autoFlush = false, $hasher = null)
    {
        $endpoint = $this->endpointRepository->findActiveByLinkAndHasher($link, $hasher);
        if ($endpoint === null) {
            $endpoint = $this->createNewForLink($link, $hasher);
            $this->entityManager->persist($endpoint);
            if ($autoFlush) {
                $this->entityManager->flush();
            }
        }
        return $endpoint;
    }

    /**
     * @param LinkInterface $link
     * @param string $hasher
     * @return EndpointEntity
     */
    public function createNewForLink(LinkInterface $link, $hasher = null)
    {
        $consumerKey = $this->generateKey();
        $privateKey = $this->generateKey();
        return new EndpointEntity($link, $consumerKey, $privateKey, $hasher);
    }

    /**
     *
     * @param LinkInterface $link
     * @return EndpointToken
     */
    public function getTokenForLink(LinkInterface $link)
    {
        $endpoint = $this->findActiveForLink($link);
        return $this->getTokenForEndpoint($endpoint);
    }

    /**
     * @param EndpointEntity $endpoint
     * @return EndpointToken
     */
    public function getTokenForEndpoint(EndpointEntity $endpoint)
    {
        foreach ($endpoint->getTokens() as $token) {
            /* @var $token EndpointToken */
            if ($token->isActive()) {
                return $token;
            }
        }
        $token = $this->generateKey();
        return $endpoint->addToken($token);
    }

    /**
     * @return string
     */
    private function generateKey()
    {
        return date("YmdHis") . sha1($this->salt . mt_rand(0, mt_getrandmax()) . '_' . time() . microtime(true));
    }
}