<?php

declare(strict_types=1);

namespace IssetBV\VideoStreamClient\Response\LiveStream\Output;

use IssetBV\VideoStreamClient\Response\LiveStream\Output\Archive\Archive;
use IssetBV\VideoStreamClient\Response\LiveStream\Output\Publish\Publish;
use IssetBV\VideoStreamClient\Response\Validator\Exception\MissingFieldsException;
use IssetBV\VideoStreamClient\Response\Validator\Validator;
use LogicException;

class Output
{
    /**
     * @var array
     */
    private $data;
    /**
     * @var Archive|null
     */
    private $archive;
    /**
     * @var Publish|null
     */
    private $publish;

    public function __construct(array $data)
    {
        self::validate($data);
        $this->data = $data;
        if (null !== $data['archive']) {
            $this->archive = new Archive($data['archive']);
        }

        if (null !== $data['publish']) {
            $this->publish = new Publish($data['publish']);
        }
    }

    public function getData(): array
    {
        return $this->data;
    }

    public function getVersion(): string
    {
        return $this->data['version'];
    }

    public function getUUID(): string
    {
        return $this->data['uuid'];
    }

    public function getType(): string
    {
        return $this->data['type'];
    }

    /**
     * @return string|null
     */
    public function getUrl()
    {
        return $this->data['url'] ?? null;
    }

    public function hasArchive(): bool
    {
        return null !== $this->archive;
    }

    public function getArchive(): Archive
    {
        if (!$this->hasArchive()) {
            throw new LogicException('no archive check with hasArchive first');
        }

        return $this->archive;
    }

    public function hasPublish(): bool
    {
        return null !== $this->publish;
    }

    public function getPublish(): Publish
    {
        if (!$this->hasPublish()) {
            throw new LogicException('no publish check with hasPublish first');
        }

        return $this->publish;
    }

    /**
     * @param array $data
     *
     * @throws MissingFieldsException
     */
    public static function validate(array $data)
    {
        Validator::checkFields($data, [
            'version',
            'uuid',
            'type',
            'archive',
            'publish',
        ]);
    }

    public static function createFromArray(array $data): self
    {
        return new self($data);
    }
}
