<?php

declare(strict_types=1);

namespace IssetBV\VideoStreamClient;

use IssetBV\VideoStreamClient\Exception\VideoStreamClientResponseException;
use IssetBV\VideoStreamClient\Request\LiveStream\Finish\FinishData;
use IssetBV\VideoStreamClient\Request\LiveStream\Quick\QuickAdd;
use IssetBV\VideoStreamClient\Response\LiveStream\LiveStream;
use IssetBV\VideoStreamClient\Response\Profile\Profile;

class VideoStreamClient
{
    /**
     * @var array
     */
    private $headers;
    /**
     * @var HttpClient
     */
    private $client;

    public function __construct(HttpClient $client, string $token)
    {
        $this->headers = [
            'x-token-auth' => $token,
            'Content-Type' => 'application/json',
        ];
        $this->client = $client;
    }

    /**
     * @param QuickAdd $quickAdd
     *
     * @throws VideoStreamClientResponseException
     *
     * @return LiveStream
     */
    public function quickPublish(QuickAdd $quickAdd): LiveStream
    {
        $body = json_encode($quickAdd);
        $response = $this->client->call('POST', '/api/v1/live-streams/quick/add', $this->headers, $body);
        if (201 !== $response->getStatusCode()) {
            throw new VideoStreamClientResponseException($response);
        }

        return LiveStream::createFromResponse($response);
    }

    /**
     * @param string $uuid
     *
     * @throws VideoStreamClientResponseException
     *
     * @return LiveStream
     */
    public function detail(string $uuid): LiveStream
    {
        $response = $this->client->call('GET', '/api/v1/live-streams/' . $uuid, $this->headers);
        if (200 !== $response->getStatusCode()) {
            throw new VideoStreamClientResponseException($response);
        }

        return LiveStream::createFromResponse($response);
    }

    /**
     * @param string $uuid
     * @param FinishData|null $finishData
     *
     * @throws VideoStreamClientResponseException
     */
    public function finish(string $uuid, FinishData $finishData = null)
    {
        if (null === $finishData) {
            $finishData = new FinishData();
        }
        $body = json_encode($finishData);
        $response = $this->client->call('PUT', "/api/v1/live-streams/{$uuid}/finish", $this->headers, $body);
        if (204 !== $response->getStatusCode()) {
            throw new VideoStreamClientResponseException($response);
        }
    }

    /**
     * @throws VideoStreamClientResponseException
     *
     * @return Profile[]
     */
    public function profiles(): array
    {
        $response = $this->client->call('GET', '/api/v1/profiles', $this->headers);
        if (200 !== $response->getStatusCode()) {
            throw new VideoStreamClientResponseException($response);
        }

        return Profile::createMultiFromResponse($response);
    }
}
