<?php

declare(strict_types=1);

namespace IssetBV\VideoPublisherClient;

use IssetBV\VideoPublisherClient\Exception\AccessDeniedException;
use IssetBV\VideoPublisherClient\Exception\FormErrorException;
use IssetBV\VideoPublisherClient\Exception\InvalidJsonBodyException;
use IssetBV\VideoPublisherClient\Exception\PublishNotFoundException;
use IssetBV\VideoPublisherClient\Exception\VideoPublisherApiErrorException;
use IssetBV\VideoPublisherClient\Request\StreamInterface;
use IssetBV\VideoPublisherClient\Response\Data\PlaylistResult;
use IssetBV\VideoPublisherClient\Response\Data\Publish;
use IssetBV\VideoPublisherClient\Response\Data\PublishListResult;
use IssetBV\VideoPublisherClient\Response\ResponseConverter;
use Psr\Http\Message\ResponseInterface;

/**
 * @author Bart Malestein <bart@isset.nl>
 * @author Tim Fennis <tim@isset.nl>
 * @author Felix Balfoor <felix@isset.nl>
 */
class VideoPublisherClient
{
    /**
     * @var array
     */
    private $headers;
    /**
     * @var VideoPublisherHttpClient
     */
    private $client;
    /**
     * @var ResponseConverter|null
     */
    private $responseConverter;

    public function __construct(VideoPublisherHttpClient $client, string $token)
    {
        $this->client = $client;
        $this->headers = [
            'x-token-auth' => $token,
            'Content-Type' => 'application/json',
        ];
    }

    /**
     * @param string $publishUuid
     *
     * @throws PublishNotFoundException
     * @throws VideoPublisherApiErrorException
     * @throws InvalidJsonBodyException
     *
     * @return Publish
     */
    public function fetchPublish(string $publishUuid): Publish
    {
        $response = $this->client->call('GET', '/api/t/v2/publishes/' . $publishUuid, $this->headers);

        $jsonData = self::decodeResponse($response);

        if (200 === $response->getStatusCode()) {
            return $this->getResponseConverter()->convertToPublish($jsonData);
        }

        if (404 === $response->getStatusCode()) {
            throw new PublishNotFoundException('Publish not found: ' . $publishUuid);
        }

        throw new VideoPublisherApiErrorException('Video publisher api responded with a error: ' . $response->getStatusCode() . ' reason: ' . $response->getReasonPhrase());
    }

    /**
     * @param int $from
     * @param int $size
     * @param string|null $search
     *
     * @throws VideoPublisherApiErrorException
     * @throws InvalidJsonBodyException
     *
     * @return PublishListResult
     */
    public function fetchPublishes(int $from = 0, int $size = 20, string $search = null): PublishListResult
    {
        $url = "/api/t/v2/publishes?from={$from}&size={$size}";
        if (null !== $search) {
            $url .= "&search={$search}";
        }

        $response = $this->client->call('GET', $url, $this->headers);
        $jsonData = self::decodeResponse($response);

        if (200 !== $response->getStatusCode()) {
            throw new VideoPublisherApiErrorException('Publishes not found');
        }

        return $this->getResponseConverter()->convertToPublishResult($jsonData);
    }

    /**
     * @param int $from
     * @param int $size
     * @param string $search
     *
     * @throws VideoPublisherApiErrorException
     * @throws InvalidJsonBodyException
     *
     * @return PlaylistResult
     */
    public function fetchPlaylists(int $from = 0, int $size = 20, string $search = null): PlaylistResult
    {
        $url = "/api/t/v2/m3u8s?from={$from}&size={$size}";
        if (null !== $search) {
            $url .= "&search={$search}";
        }

        $response = $this->client->call('GET', $url, $this->headers);

        $jsonData = self::decodeResponse($response);

        if (200 !== $response->getStatusCode()) {
            throw new VideoPublisherApiErrorException('M3U8s not found');
        }

        return $this->getResponseConverter()->convertToPlaylistResult($jsonData);
    }

    /**
     * @param StreamInterface $stream
     *
     * @throws FormErrorException
     * @throws InvalidJsonBodyException
     * @throws VideoPublisherApiErrorException
     *
     * @return Publish
     */
    public function publish(StreamInterface $stream): Publish
    {
        $publish = [
            'stream_name' => $stream->getFilename(),
            'callback_url' => $stream->getCallbackUrl(),
            'identifier' => $stream->getIdentifier(),
            'description' => $stream->getDescription(),
            'locations' => [],
        ];

        foreach ($stream->getExtraField() as $extraField) {
            $publish[$extraField->getName()] = $extraField->getValue();
        }

        foreach ($stream->getFiles() as $file) {
            $publish['locations'][] = $file->getLocationData();
        }

        $response = $this->client->call('POST', '/api/t/v2/publishes', $this->headers, json_encode(['publish' => $publish]));
        $jsonData = self::decodeResponse($response);

        if (400 === $response->getStatusCode()) {
            throw new FormErrorException($response, 'Form error');
        }

        if (200 !== $response->getStatusCode()) {
            throw new VideoPublisherApiErrorException('Not created error');
        }

        return $this->getResponseConverter()->convertToPublish($jsonData);
    }

    /**
     * @return ResponseConverter
     */
    private function getResponseConverter(): ResponseConverter
    {
        if (null === $this->responseConverter) {
            $this->responseConverter = new ResponseConverter();
        }

        return $this->responseConverter;
    }

    /**
     * @param ResponseInterface $response
     *
     * @throws InvalidJsonBodyException
     * @throws AccessDeniedException
     *
     * @return mixed
     */
    private static function decodeResponse(ResponseInterface $response)
    {
        if (401 === $response->getStatusCode() || 403 === $response->getStatusCode()) {
            throw new AccessDeniedException($response, 'access denied');
        }
        $body = (string) $response->getBody();

        $decoded = \json_decode($body, true);

        if (JSON_ERROR_NONE !== \json_last_error()) {
            throw new InvalidJsonBodyException('Json Exception: ' . \json_last_error_msg());
        }

        return $decoded;
    }
}
