<?php

namespace IssetBV\VideoPublisherClient;

use DateTime;
use IssetBV\VideoPublisherClient\Exception\InvalidJsonBodyException;
use IssetBV\VideoPublisherClient\Exception\NegativeStatusCodeException;
use IssetBV\VideoPublisherClient\Exception\ResponseException;
use IssetBV\XAuthClient\Client;
use IssetBV\XAuthClient\Connection\ConnectionInterface;
use IssetBV\XAuthClient\Connection\ResponseInterface;
use IssetBV\XAuthClient\Payload\Payload;
use IssetBV\XAuthClient\Security\Token;
use League\Uri\UriParser;
use stdClass;

/**
 * @author Bart Malestein <bart@isset.nl>
 * @author Tim Fennis <tim@isset.nl>
 */
class VideoPublisherClient extends Client
{
    /**
     * @var string
     */
    private $url;

    /**
     * @var Token
     */
    private $token;

    /**
     * VideoPublisherClient constructor.
     *
     * @param ConnectionInterface $connection
     * @param Token $token
     * @param string $url
     */
    public function __construct(ConnectionInterface $connection, Token $token, string $url)
    {
        parent::__construct($connection);
        $this->token = $token;
        $this->url = rtrim($url, '/');
    }

    /**
     * @param string $publishUuid
     *
     * @throws ResponseException
     *
     * @return Publish
     */
    public function fetchPublishData($publishUuid)
    {
        $response = $this->sendPayloadWithToken($this->createPayload('GET', '/v2/publishes/' . $publishUuid), $this->token);
        $jsonData = self::decodeResponse($response);

        return $this->convertJsonToPublish($jsonData);
    }

    /**
     * @param int $from
     * @param int $size
     * @param string $search
     *
     * @return PublishResult
     *
     * @throws InvalidJsonBodyException
     * @throws NegativeStatusCodeException
     */
    public function fetchPublishes($from = 0, $size = 20, $search = '')
    {
        $response = $this->sendPayloadWithToken($this->createPayload('GET', "/v2/publishes?from={$from}&size={$size}&search={$search}"), $this->token);
        $jsonData = self::decodeResponse($response);

        return new PublishResult($jsonData->total, $jsonData->from, $jsonData->size, array_map([$this, 'convertJsonToPublish'], $jsonData->results));
    }

    /**
     * @throws InvalidJsonBodyException
     * @throws NegativeStatusCodeException
     *
     * @return Playlist[]
     */
    public function fetchPlaylists()
    {
        $response = $this->sendPayloadWithToken($this->createPayload('GET', '/m3u8'), $this->token);
        $jsonData = self::decodeResponse($response);

        return array_map([$this, 'convertJsonToPlaylist'], $jsonData);
    }

    /**
     * @param Stream $stream
     *
     * @throws ResponseException
     *
     * @return string The uuid of the publish just created
     */
    public function publish(Stream $stream)
    {
        $data['publish']['stream_name'] = $stream->getFilename();
        $data['publish']['callback_url'] = null;
        if ($stream->getIdentifier()) {
            $data['publish']['identifier'] = $stream->getIdentifier();
        }
        if ($stream->getVideoArchiveUuid()) {
            $data['publish']['video_archive_uuid'] = $stream->getVideoArchiveUuid();
        }
        if ($stream->getDescription()) {
            $data['publish']['description'] = $stream->getDescription();
        }

        $parser = new UriParser();
        if ($stream->getImage()) {
            $parts = $parser->parse($stream->getImage()->getUrl());
            $data['publish']['locations'][] = [
                'location' => $parts['host'],
                'path' => ltrim($parts['path'], '/'),
                'user' => $parts['user'],
                'password' => $parts['pass'],
                'type' => 'ftp',
                'import_type' => 'image'
            ];
        }

        foreach ($stream->getFiles() as $file) {
            $parts = $parser->parse($file->getUrl());

            $data['publish']['locations'][] = [
                'location' => $parts['host'],
                'path' => ltrim($parts['path'], '/'),
                'user' => $parts['user'],
                'password' => $parts['pass'],
                'type' => 'ftp',
            ];
        }

        $response = $this->sendPayloadWithToken($this->createPayload('post', '/v2/publishes', $data), $this->token);
        $jsonContent = self::decodeResponse($response);

        if (false === isset($jsonContent->uuid)) {
            throw new ResponseException('Json response did not contain uuid property');
        }

        return $jsonContent->uuid;
    }

    /**
     * @param stdClass $jsonData
     *
     * @return Publish
     */
    private function convertJsonToPublish($jsonData)
    {
        return new Publish(
            $jsonData->uuid,
            $jsonData->stream_name,
            $jsonData->status,
            $jsonData->enabled,
            new DateTime($jsonData->date_created),
            property_exists($jsonData, 'viewable') ? $jsonData->viewable : false,
            property_exists($jsonData, 'view') ? new View(
                $jsonData->view->playout_url,
                $jsonData->view->video_player
            ) : null,
            $jsonData->identifier,
            $jsonData->description,
            array_map(function($asset){
                return new Asset($asset->url, $asset->type, $asset->status);
            }, $jsonData->assets)
        );
    }

    /**
     * @param stdClass $jsonData
     *
     * @return Publish
     */
    private function convertJsonToPlaylist($jsonData)
    {
        return new Playlist(
            $jsonData->name,
            $jsonData->url
        );
    }

    /**
     * @param string $method pOST, GET, PUT, HEAD etc
     * @param string $path
     * @param null|array $data
     *
     * @return Payload
     */
    private function createPayload($method, $path, $data = null)
    {
        $payload = new Payload($this->url . $path, $method);

        $payload->setHeader('Content-Type', 'application/json');
        $payload->setJson(true);
        if ($data !== null) {
            $payload->overwritePostData($data);
        }

        return $payload;
    }

    /**
     * @param ResponseInterface $response
     *
     * @throws ResponseException
     *
     * @return stdClass
     */
    private static function decodeResponse(ResponseInterface $response)
    {
        $code = $response->getStatusCode();
        if ($code < 200 || $code > 299) {
            throw new NegativeStatusCodeException($response->getStatusCode() . ': ' . $response->getContent());
        }

        $decoded = json_decode($response->getContent());

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidJsonBodyException('Json Exception: ' . json_last_error_msg());
        }

        return $decoded;
    }
}
