<?php

namespace IssetBV\VideoPublisherClient;

use DateTime;
use IssetBV\VideoPublisherClient\Exception\InvalidJsonBodyException;
use IssetBV\VideoPublisherClient\Exception\NegativeStatusCodeException;
use IssetBV\VideoPublisherClient\Exception\ResponseException;
use IssetBV\XAuthClient\Client;
use IssetBV\XAuthClient\Connection\ConnectionInterface;
use IssetBV\XAuthClient\Connection\ResponseInterface;
use IssetBV\XAuthClient\Payload\Payload;
use IssetBV\XAuthClient\Security\Token;
use League\Uri\UriParser;
use stdClass;

/**
 * @author Bart Malestein <bart@isset.nl>
 * @author Tim Fennis <tim@isset.nl>
 */
class VideoPublisherClient extends Client
{
    /**
     * @var string
     */
    private $url;

    /**
     * @var Token
     */
    private $token;

    /**
     * VideoPublisherClient constructor.
     *
     * @param ConnectionInterface $connection
     * @param Token $token
     * @param string $url
     */
    public function __construct(ConnectionInterface $connection, Token $token, string $url)
    {
        parent::__construct($connection);
        $this->token = $token;
        $this->url = rtrim($url, '/');
    }

    /**
     * @param string $publishUuid
     *
     * @throws ResponseException
     *
     * @return Publish
     */
    public function fetchPublishData($publishUuid)
    {
        $response = $this->sendPayloadWithToken($this->createPayload('GET', '/publish/' . $publishUuid), $this->token);
        $jsonData = self::decodeResponse($response);

        return $this->convertJsonToPublish($jsonData);
    }

    /**
     * @throws ResponseException
     *
     * @return Publish[]
     */
    public function fetchAll()
    {
        $response = $this->sendPayloadWithToken($this->createPayload('GET', '/published'), $this->token);
        $jsonData = self::decodeResponse($response);

        return array_map([$this, 'convertJsonToPublish'], $jsonData);
    }

    /**
     * @param Stream $stream
     *
     * @throws ResponseException
     *
     * @return string The uuid of the publish just created
     */
    public function publish(Stream $stream)
    {
        $data['publish']['stream_name'] = $stream->getFilename();
        $data['publish']['callback_url'] = null;

        $parser = new UriParser();

        foreach ($stream->getFiles() as $file) {
            $parts = $parser->parse($file->getUrl());

            $data['publish']['locations'][] = [
                'location' => $parts['host'],
                'path' => ltrim($parts['path'], '/'),
                'user' => $parts['user'],
                'password' => $parts['pass'],
                'type' => 'ftp',
            ];
        }

        $response = $this->sendPayloadWithToken($this->createPayload('post', '/publish', $data), $this->token);
        $jsonContent = self::decodeResponse($response);

        if (false === isset($jsonContent->uuid)) {
            throw new ResponseException('Json response did not contain uuid property');
        }

        return $jsonContent->uuid;
    }

    /**
     * @param stdClass $jsonData
     *
     * @return Publish
     */
    private function convertJsonToPublish($jsonData)
    {
        return new Publish(
            $jsonData->uuid,
            $jsonData->status,
            $jsonData->enabled,
            new DateTime($jsonData->date_created),
            property_exists($jsonData, 'viewable') ? $jsonData->viewable : false,
            property_exists($jsonData, 'view') ? new View(
                $jsonData->view->playout_url,
                $jsonData->view->video_player
            ) : null
        );
    }

    /**
     * @param string $method pOST, GET, PUT, HEAD etc
     * @param string $path
     * @param null|array $data
     *
     * @return Payload
     */
    private function createPayload($method, $path, $data = null)
    {
        $payload = new Payload($this->url . $path, $method);

        $payload->setHeader('Content-Type', 'application/json');
        $payload->setJson(true);
        if ($data !== null) {
            $payload->overwritePostData($data);
        }

        return $payload;
    }

    /**
     * @param ResponseInterface $response
     *
     * @throws ResponseException
     *
     * @return stdClass
     */
    private static function decodeResponse(ResponseInterface $response)
    {
        $code = $response->getStatusCode();
        if ($code < 200 || $code > 299) {
            throw new NegativeStatusCodeException($response->getStatusCode() . ': ' . $response->getContent());
        }

        $decoded = json_decode($response->getContent());

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidJsonBodyException('Json Exception: ' . json_last_error_msg());
        }

        return $decoded;
    }
}
