<?php

namespace IssetBV\VideoPublisherClient;

use IssetBV\VideoPublisherClient\Exception\InvalidJsonBodyException;
use IssetBV\VideoPublisherClient\Exception\NegativeStatusCodeException;
use IssetBV\VideoPublisherClient\Exception\ResponseException;
use IssetBV\XAuthClient\Client;
use IssetBV\XAuthClient\Connection\ConnectionInterface;
use IssetBV\XAuthClient\Connection\ResponseInterface;
use IssetBV\XAuthClient\Payload\Payload;
use IssetBV\XAuthClient\Security\Token;
use League\Uri\Schemes\Http;
use League\Uri\UriParser;

/**
 * Class VideoPublisherConnector
 *
 * @package IssetBV\Niop\AssetBundle\Service
 * @author Bart Malestein <bart@isset.nl>
 * @author Tim Fennis <tim@isset.nl>
 */
class VideoPublisherClient extends Client
{

    /**
     * @var string
     */
    private $url;

    /**
     * @var Token
     */
    private $token;

    /**
     * VideoPublisherClient constructor.
     * @param ConnectionInterface $connection
     * @param Token $token
     * @param string $url
     */
    public function __construct(ConnectionInterface $connection, Token $token, string $url)
    {
        parent::__construct($connection);
        $this->token = $token;
        $this->url = rtrim($url, '/');
    }

    /**
     * @param string $publishUuid
     * @return Publish
     * @throws \IssetBV\VideoPublisherClient\Exception\ResponseException
     */
    public function fetchPublishData($publishUuid)
    {
        $response = $this->sendPayloadWithToken($this->createPayload('get', '/publish/' . $publishUuid), $this->token);
        $jsonData = self::decodeResponse($response);

        return new Publish(
            $jsonData->uuid,
            $jsonData->status,
            $jsonData->enabled,
            $jsonData->viewable,
            new View(
                $jsonData->view->playout_url,
                $jsonData->view->video_player
            )
        );
    }

    /**
     * @param Stream $stream
     * @throws ResponseException
     *
     * @return string The uuid of the publish just created
     */
    public function publish(Stream $stream)
    {
        $data = [
            'publish[stream_name]' => $stream->getFilename(), // playable filename
            'publish[callback_url]' => null
        ];

        $fileCount = 0;
        $parser = new UriParser();
        foreach ($stream->getFiles() as $file) {

            $parts = $parser->parse($file->getUrl());

            $data['publish[locations][' . $fileCount . '][location]'] = $parts['host'];
            $data['publish[locations][' . $fileCount . '][path]'] = ltrim($parts['path'], '/');
            $data['publish[locations][' . $fileCount . '][user]'] = $parts['user'];
            $data['publish[locations][' . $fileCount . '][password]'] = $parts['pass'];
            $data['publish[locations][' . $fileCount . '][type]'] = 'ftp';

            ++$fileCount;
        }

        $response = $this->sendPayloadWithToken($this->createPayload('post', '/publish', $data), $this->token);
        $jsonContent = self::decodeResponse($response);

        if (false === isset($jsonContent->uuid)) {
            throw new ResponseException('Json response did not contain uuid property');
        }

        return $jsonContent->uuid;
    }

    /**
     * @param string $method POST, GET, PUT, HEAD etc.
     * @param string $path
     * @param null|array $data
     * @return Payload
     */
    protected function createPayload($method, $path, $data = null)
    {
        $payload = new Payload($this->url . $path, $method);
        $payload->setHeader('Content-Type', 'application/json');
        if ($data !== null) {
            $payload->overwritePostData($data);
        }

        return $payload;
    }

    /**
     * @param ResponseInterface $response
     * @return string
     * @throws ResponseException
     */
    protected static function decodeResponse(ResponseInterface $response)
    {
        dump($response->getJsonResponse());
        die;
        if ($response->getStatusCode() < 200 || $response->getStatusCode() > 299) {
            throw new NegativeStatusCodeException($response->getStatusCode());
        }

        $decoded = json_decode($response->getContent());

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidJsonBodyException('Json Exception: ' . json_last_error_msg());
        }

        return $decoded;
    }

}