<?php

namespace IssetBV\VideoPublisherClient;

use IssetBV\VideoPublisherClient\Exception\InvalidJsonBodyException;
use IssetBV\VideoPublisherClient\Exception\NegativeStatusCodeException;
use IssetBV\VideoPublisherClient\Exception\ResponseException;
use IssetBV\XAuthClient\Connection\ConnectionInterface;
use IssetBV\XAuthClient\Connection\ResponseInterface;
use IssetBV\XAuthClient\Payload\Payload;
use League\Uri\Schemes\Http;
use League\Uri\UriParser;

/**
 * Class VideoPublisherConnector
 *
 * @package IssetBV\Niop\AssetBundle\Service
 * @author Bart Malestein <bart@isset.nl>
 * @author Tim Fennis <tim@isset.nl>
 */
class VideoPublisherClient
{
    /**
     * @var ConnectionInterface
     */
    private $connection;

    /**
     * @var Http
     */
    private $httpUrl;

    /**
     * @var string
     */
    private $xAuthToken;

    /**
     * @param ConnectionInterface $connection
     * @param string $xAuthToken
     * @param string $serverHostname
     */
    public function __construct(ConnectionInterface $connection, $xAuthToken, $serverHostname)
    {
        $this->connection = $connection;
        $this->xAuthToken = $xAuthToken;

        $this->httpUrl = Http::createFromComponents([
            'scheme' => 'http',
            'host' => $serverHostname,
            'port' => 80,
            'path' => '/',
        ]);
    }

    /**
     * @param string $publishUuid
     * @return Publish
     */
    public function fetchPublishData($publishUuid)
    {
        $response = $this->sendPayload($this->createPayload('get', 'api/publish/' . $publishUuid));
        $jsonData = self::decodeResponse($response);

        return new Publish(
            $jsonData->uuid,
            $jsonData->status,
            $jsonData->enabled,
            $jsonData->viewable,
            new View(
                $jsonData->view->playout_url,
                $jsonData->view->video_player
            )
        );
    }

    /**
     * @param Stream $stream
     * @throws ResponseException
     *
     * @return string The uuid of the publish just created
     */
    public function publish(Stream $stream)
    {
        $data = [
            'publish[stream_name]' => $stream->getFilename(), // playable filename
            'publish[callback_url]' => null
        ];

        $fileCount = 0;
        foreach ($stream->getFiles() as $file) {
            $parser = new UriParser();
            $parts = $parser->parse($file->getUrl());

            $data['publish[locations][' . $fileCount . '][location]'] = $parts['host'];
            $data['publish[locations][' . $fileCount . '][path]'] = ltrim($parts['path'], '/');
            $data['publish[locations][' . $fileCount . '][user]'] = $parts['user'];
            $data['publish[locations][' . $fileCount . '][password]'] = $parts['pass'];
            $data['publish[locations][' . $fileCount . '][type]'] = 'ftp';

            ++$fileCount;
        }

        $response = $this->sendPayload($this->createPayload('post', 'api/publish', $data));
        $jsonContent = self::decodeResponse($response);

        if (false === isset($jsonContent->uuid)) {
            throw new ResponseException('Json response did not contain uuid property');
        }

        return $jsonContent->uuid;
    }

    /**
     * @param string $method POST, GET, PUT, HEAD etc.
     * @param string $path
     * @param null|array $data
     * @return Payload
     */
    protected function createPayload($method, $path, $data = null)
    {
        $url = $this->httpUrl->withPath($path);

        $payload = new Payload($url->__toString(), $method);
//        $payload->setHeader('Content-Type', 'application/json');
        $payload->setHeader('xauth-token', $this->xAuthToken);

        if ($data !== null) {
            $payload->overwritePostData($data);
        }

        return $payload;
    }

    /**
     * @param Payload $payload
     * @return ResponseInterface
     */
    protected function sendPayload(Payload $payload)
    {
        return $this->connection->sendPayload($payload);
    }

    /**
     *
     *
     * @param ResponseInterface $response
     * @return string
     * @throws ResponseException
     */
    protected static function decodeResponse(ResponseInterface $response)
    {
        if ($response->getStatusCode() < 200 || $response->getStatusCode() > 299) {
            throw new NegativeStatusCodeException();
        }

        $decoded = json_decode($response->getContent());

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidJsonBodyException('Json Exception: ' . json_last_error_msg());
        }

        return $decoded;
    }

}