<?php

declare(strict_types=1);

namespace IssetBV\VideoArchiveClient;

use IssetBV\VideoArchiveClient\Exception\VideoArchiveResponseException;
use IssetBV\VideoArchiveClient\Request\File\Export\ExportInterface;
use IssetBV\VideoArchiveClient\Request\File\File;
use IssetBV\VideoArchiveClient\Response\File\FileDetail;
use IssetBV\VideoArchiveClient\Response\File\FileSearch;
use IssetBV\VideoArchiveClient\Response\Folder\Folder;
use IssetBV\VideoArchiveClient\Response\Folder\FolderData;
use IssetBV\VideoArchiveClient\Response\Job\JobData;

class VideoArchiveClient
{
    /**
     * @var array
     */
    private $headers;
    /**
     * @var VideoArchiveHttpClient
     */
    private $client;

    public function __construct(VideoArchiveHttpClient $client, string $token)
    {
        $this->client = $client;
        $this->headers = [
            'x-token-auth' => $token,
            'Content-Type' => 'application/json',
        ];
    }

    /**
     * @param string $search
     * @param array $tags
     * @param array $meta
     *
     * @throws VideoArchiveResponseException
     *
     * @return FileSearch[]
     */
    public function fileSearch(string $search, array $tags = [], array $meta = []): array
    {
        $query = [
            'q' => $search,
            'tags' => $tags,
            'meta' => $meta,
        ];

        $response = $this->client->call('GET', '/api/v1/files', $this->headers, null, $query);
        if (200 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'Search failed');
        }

        return FileSearch::createMultiFromResponse($response);
    }

    /**
     * @param File $file
     *
     * @throws VideoArchiveResponseException
     *
     * @return FileDetail
     */
    public function fileCreateFtp(File $file): FileDetail
    {
        $response = $this->client->call('POST', '/api/v1/files/add/ftp', $this->headers, json_encode($file));
        if (201 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'file not created');
        }

        return FileDetail::createFromResponse($response);
    }

    /**
     * @param string $uuid
     *
     * @throws VideoArchiveResponseException
     *
     * @return FileDetail
     */
    public function fileDetails(string $uuid): FileDetail
    {
        $response = $this->client->call('GET', "/api/v1/files/{$uuid}", $this->headers);
        if (200 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'file not found');
        }

        return FileDetail::createFromResponse($response);
    }

    /**
     * @param string $uuid
     * @param ExportInterface $data
     *
     * @throws VideoArchiveResponseException
     *
     * @return JobData
     */
    public function fileExport(string $uuid, ExportInterface $data): JobData
    {
        $response = $this->client->call('POST', "/api/v1/files/{$uuid}/export", $this->headers, json_encode($data));

        if (201 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'file not exported');
        }

        return JobData::createFromResponse($response);
    }

    public function folderCreate(string $path, string $parentUUID = null)
    {
        $data = json_encode([
            'path' => $path,
            'folder_base_uuid' => $parentUUID,
        ]);

        $response = $this->client->call('POST', '/api/v1/folders', $this->headers, $data);

        if (201 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'file not exported');
        }

        return Folder::createFromResponse($response);
    }

    public function folders(): FolderData
    {
        $response = $this->client->call('GET', '/api/v1/folders', $this->headers);

        if (200 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'file not exported');
        }

        return FolderData::createFromResponse($response);
    }

    public function folderDetail(string $uuid)
    {
        $response = $this->client->call('GET', "/api/v1/folders/{$uuid}", $this->headers);

        if (200 !== $response->getStatusCode()) {
            throw new VideoArchiveResponseException($response, 'file not exported');
        }

        return Folder::createFromResponse($response);
    }
}
