<?php

declare(strict_types=1);

namespace IssetBV\VideoArchiveClient\Response\File;

use DateTime;
use Psr\Http\Message\ResponseInterface;

class FileDetail
{
    /**
     * @var array
     */
    private $data;
    /**
     * @var string
     */
    private $uuid;
    /**
     * @var string
     */
    private $name;
    /**
     * @var string
     */
    private $state;
    /**
     * @var int|null
     */
    private $size;
    /**
     * @var DateTime
     */
    private $created;
    /**
     * @var DateTime
     */
    private $updated;
    /**
     * @var float|null
     */
    private $duration;
    /**
     * @var string|null
     */
    private $extension;
    /**
     * @var string
     */
    private $owner;
    /**
     * @var string|null
     */
    private $mimeType;
    /**
     * @var FileMeta[]
     */
    private $meta;
    /**
     * @var FileTag[]
     */
    private $tags;
    /**
     * @var FileData[]
     */
    private $extra;
    /**
     * @var FileData|null
     */
    private $defaultImage;
    /**
     * @var FileData|null
     */
    private $defaultPreview;
    /**
     * @var string
     */
    private $creator;
    /**
     * @var FileFolderData
     */
    private $folder;

    public function __construct(array $data)
    {
        $this->data = $data;
        $this->uuid = $data['uuid'];
        $this->name = $data['name'];
        $this->owner = $data['owner'];
        $this->state = $data['state'];
        $this->mimeType = $data['mime_type'];
        $this->size = null === $data['size'] ? null : (int) $data['size'];
        $this->folder = new FileFolderData($data['folder']);
        $this->meta = [];
        foreach ($data['meta'] as $meta) {
            $this->meta[] = new FileMeta($meta);
        }
        $this->tags = [];
        foreach ($data['tags'] as $tag) {
            $this->tags[] = new FileTag($tag);
        }

        $this->defaultImage = null === $data['default_image'] ? null : new FileData($data['default_image']);
        $this->defaultPreview = null === $data['default_preview'] ? null : new FileData($data['default_preview']);
        $this->duration = null === $data['duration'] ? null : (float) $data['duration'];
        $this->extra = [];

        foreach ($data['extra'] as $extra) {
            $this->extra[] = new FileData($extra);
        }
        $this->created = new DateTime($data['created']);
        $this->updated = new DateTime($data['updated']);
        $this->creator = $data['creator'];
        $this->extension = $data['extension'];
    }

    /**
     * @return array
     */
    public function getData(): array
    {
        return $this->data;
    }

    /**
     * @return string
     */
    public function getUuid(): string
    {
        return $this->uuid;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getState(): string
    {
        return $this->state;
    }

    /**
     * @return int|null
     */
    public function getSize()
    {
        return $this->size;
    }

    /**
     * @return FileFolderData
     */
    public function getFolder(): FileFolderData
    {
        return $this->folder;
    }

    /**
     * @return DateTime
     */
    public function getCreated(): DateTime
    {
        return $this->created;
    }

    /**
     * @return DateTime
     */
    public function getUpdated(): DateTime
    {
        return $this->updated;
    }

    /**
     * @return float|null
     */
    public function getDuration()
    {
        return $this->duration;
    }

    /**
     * @return null|string
     */
    public function getExtension()
    {
        return $this->extension;
    }

    /**
     * @return string
     */
    public function getOwner(): string
    {
        return $this->owner;
    }

    /**
     * @return null|string
     */
    public function getMimeType()
    {
        return $this->mimeType;
    }

    /**
     * @return FileMeta[]
     */
    public function getMeta(): array
    {
        return $this->meta;
    }

    /**
     * @return FileTag[]
     */
    public function getTags(): array
    {
        return $this->tags;
    }

    /**
     * @return FileData[]
     */
    public function getExtra(): array
    {
        return $this->extra;
    }

    /**
     * @return FileData|null
     */
    public function getDefaultImage()
    {
        return $this->defaultImage;
    }

    /**
     * @return FileData|null
     */
    public function getDefaultPreview()
    {
        return $this->defaultPreview;
    }

    /**
     * @return string
     */
    public function getCreator(): string
    {
        return $this->creator;
    }

    public static function createFromResponse(ResponseInterface $response): self
    {
        $data = json_decode((string) $response->getBody(), true);

        return new self($data);
    }
}
