<?php

namespace IssetBV\VideoArchiveClient;

use IssetBV\VideoArchiveClient\Security\TokenCreator;
use LogicException;
use IssetBV\VideoArchiveClient\Security\TokenVault;
use IssetBV\VideoArchiveClient\Payload\Payload;
use IssetBV\VideoArchiveClient\Connection\ConnectionInterface;
use IssetBV\VideoArchiveClient\Connection\ResponseInterface;

/**
 * Class VideoArchiveSecuredClient.
 *
 * @author Felix Balfoort <felix@isset.nl>
 */
class VideoArchiveSecuredClient
{
    /**
     * @var TokenCreator
     */
    private $tokenCreator;

    /**
     * @var TokenVault
     */
    private $tokenVault;

    /**
     * @var ConnectionInterface
     */
    private $connection;

    /**
     * @var string
     */
    private $consumerKey;

    /**
     * @var string
     */
    private $privateKey;

    /**
     * VideoArchiveSecuredClient constructor.
     *
     * @param ConnectionInterface $connectionInterface
     * @param TokenCreator $tokenCreator
     * @param TokenVault $tokenVault
     * @param string $consumerKey
     * @param string $privateKey
     */
    public function __construct(ConnectionInterface $connectionInterface, TokenCreator $tokenCreator, TokenVault $tokenVault, $consumerKey, $privateKey)
    {
        $this->connection = $connectionInterface;
        $this->tokenCreator = $tokenCreator;
        $this->tokenVault = $tokenVault;
        $this->consumerKey = $consumerKey;
        $this->privateKey = $privateKey;
    }

    /**
     * Adds the security and Content-Type headers to the payload before sending them.
     *
     * @param Payload $payload
     * @param bool $retry
     *
     * @return ResponseInterface
     */
    public function sendPayloadWithHeaders(Payload $payload, $retry = true)
    {
        $this->setRequiredHeadersToPayload($payload);
        $response = $this->sendPayload($payload);
        if ($retry === true && $response->getStatusCode() === 403) {
            $this->tokenVault->removeToken($this->consumerKey);

            return $this->sendPayloadWithHeaders($payload, false);
        }

        return $response;
    }

    /**
     * Set the content-type and xauth-token header to a payload.
     *
     * @param Payload $payload
     */
    public function setRequiredHeadersToPayload(Payload $payload)
    {
        $payload->setHeader('Content-Type', 'application/json');
        $payload->setHeader('xauth-token', $this->getToken());
    }

    public function getToken()
    {
        try {
            $token = $this->tokenVault->getToken($this->consumerKey);
        } catch (LogicException $e) {
            $token = $this->tokenCreator->getToken($this->consumerKey, $this->privateKey);
            $this->tokenVault->addToken($this->consumerKey, $token);
        }

        return $token;
    }

    /**
     * @param Payload $payload
     *
     * @return ResponseInterface
     */
    public function sendPayload(Payload $payload)
    {
        return $this->connection->sendPayload($payload);
    }
}
