<?php
namespace IssetBV\VideoArchiveClient;

use LogicException;
use IssetBV\VideoArchiveClient\Security\TokenCreater;
use IssetBV\VideoArchiveClient\Security\TokenVault;
use IssetBV\VideoArchiveClient\Payload\Payload;
use IssetBV\VideoArchiveClient\Connection\ConnectionInterface;
use IssetBV\VideoArchiveClient\Connection\ResponseException;

class VideoArchiveSecuredClient
{

    /**
     * 
     * @var TokenCreater
     */
    private $tokenCreater;

    /**
     * @var TokenVault
     */
    private $tokenVault;

    /**
     * @var ConnectionInterface
     */
    private $connection;

    private $consumerKey;

    private $privateKey;

    private public function __construct(ConnectionInterface $connectionInterface, TokenCreater $tokenCreater, TokenVault $tokenVault, $consumerKey, $privateKey)
    {
        $this->connection = $connectionInterface;
        $this->tokenCreater = $tokenCreater;
        $this->tokenVault = $tokenVault;
        $this->consumerKey = $consumerKey;
        $this->privateKey = $privateKey;
    }

    /**
     * Adds the security and Content-Type headers to the payload before sending them
     *
     * @param Payload $payload
     * @return ResponseInterface
     */
    public function sendPayloadWithHeaders(Payload $payload, $reTry = true)
    {
        try {
            $this->setRequiredHeadersToPayload($payload);
            return $this->sendPayload($payload);
        } catch (ResponseException $e) {
            if ($e->getResponse()->getStatusCode() == 403 && $reTry) {
                $this->tokenVault->removeToken($this->consumerKey);
                return $this->sendPayloadWithHeaders($payload, false);
            } else {
                throw $e;
            }
        }
    }

    /**
     * set the content-type and xauth-token header to a payload
     *
     * @param Payload $payload
     */
    public function setRequiredHeadersToPayload(Payload $payload)
    {
        $payload->setHeader('Content-Type', 'application/json');
        $payload->setHeader('xauth-token', $this->getToken());
    }

    public function getToken()
    {
        try {
            $token = $this->tokenVault->getToken($this->consumerKey);
        } catch (LogicException $e) {
            $token = $this->tokenCreater->getToken($this->consumerKey, $this->privateKey);
            $this->tokenVault->addToken($this->consumerKey, $token);
        }
        return $token;
    }

    /**
     * @param Payload $payload
     * @return ResponseInterface
     */
    public function sendPayload(Payload $payload)
    {
        return $this->connection->sendPayload($payload);
    }
}