<?php
namespace IssetBV\FileHandlerClient\File;

use DateTime;
use GuzzleHttp\Client as GuzzleClient;
use IssetBV\FileHandlerClient\File\Location\LocationInterface;

class Files
{

    /**
     * @var GuzzleClient
     */
    private $client;

    /**
     * 
     * @var array
     */
    private $defaultDivisions;

    /**
     * @param GuzzleClient $client
     * @param array $defaultDivisions
     */
    public function __construct(GuzzleClient $client, array $defaultDivisions = array())
    {
        $this->defaultDivisions = $defaultDivisions;
        $this->client = $client;
    }

    public function getFailed()
    {
        $data = json_decode($this->client->get('files/failed')
            ->getBody()
            ->getContents());
        $return = array();
        foreach ($data as $fileData) {
            $return[] = new File($this->client, $fileData);
        }
        return $return;
    }

    public function getUsage()
    {
        $data = $this->client->get('file/usage')
            ->getBody()
            ->getContents();
        
        return json_decode($data);
    }

    public function getShare($token)
    {
        $data = $this->client->get('file/share/' . $token)
            ->getBody()
            ->getContents();
        
        return json_decode($data);
    }

    /**
     * 
     * @param string $query
     * @param array $tags
     * @return multitype:\IssetBV\FileHandlerClient\File\File
     */
    public function search($query, array $tags = null)
    {
        $query = [
            'q' => $query
        ];
        if (! empty($tags) && is_array($tags)) {
            $query['tags'] = $tags;
        }
        
        $data = $this->client->get('file/search', [
            'query' => $query
        ])
            ->getBody()
            ->getContents();
        return json_decode($data);
    }

    /**
     * @param string $uuid
     * @return \IssetBV\FileHandlerClient\File\File
     */
    public function get($uuid)
    {
        $data = $this->client->get('file/' . $uuid)
            ->getBody()
            ->getContents();
        return new File($this->client, $data);
    }

    public function multi(array $files)
    {
        $data = json_decode($this->client->get('files/multi', [
            'query' => [
                'files' => $files
            ]
        ])
            ->getBody()
            ->getContents());
        
        $return = [
            'files' => [],
            'errors' => $data->errors
        ];
        foreach ($data->files as $file) {
            $return['files'][] = new File($this->client, $file);
        }
        
        return $return;
    }

    /**
     * 
     * @param string $name
     * @param LocationInterface $locationSource
     * @param string $folderUuid
     * @return \IssetBV\FileHandlerClient\File\File
     */
    public function add($name, LocationInterface $locationSource, $folderUuid = null, $preview = true, $callbackUrl = null)
    {
        $data = $this->client->post('file', [
            'json' => [
                'name' => $name,
                'location_source' => $locationSource->toArray(),
                'folder_uuid' => $folderUuid,
                'preview' => $preview === false ? 0 : $preview,
                'callback_url' => $callbackUrl
            ]
        ])
            ->getBody()
            ->getContents();
        return new File($this->client, $data);
    }

    public function recent(DateTime $after = null)
    {
        if ($after === null) {
            $after = new DateTime('-3 days');
        }
        $data = json_decode($this->client->get('files/recent', [
            'query' => [
                'after_date' => $after->format('c')
            ]
        ])
            ->getBody()
            ->getContents());
        return $data;
    }

    public function tagCloud()
    {
        $data = json_decode($this->client->get('tag/cloud')
            ->getBody()
            ->getContents());
        return $data;
    }
}