<?php
namespace IssetBV\FileHandlerClient\File;

use DateTime;
use GuzzleHttp\Client as GuzzleClient;
use IssetBV\FileHandlerClient\Folder\Folder;
use IssetBV\FileHandlerClient\File\Location\LocationInterface;

class File
{

    /**
     * @var GuzzleClient
     */
    private $client;

    private $uuid;

    private $name;

    private $owner;

    private $state;

    private $mimeType;

    private $size;

    /**
     * @var Folder
     */
    private $folder;

    private $meta;

    private $tags;

    private $extra;

    private $extension;

    /**
     * 
     * @var DateTime
     */
    private $created;

    /**
     *
     * @var DateTime
     */
    private $updated;

    private $probe = false;

    private $creator;

    private $defaultImage;

    private $defaultPreview;

    private $duration;

    public function __construct($client, $data)
    {
        $this->client = $client;
        $this->setFromData($data);
    }

    private function setFromData($data)
    {
        if (is_string($data)) {
            $data = json_decode($data);
        }
        $this->uuid = $data->uuid;
        $this->name = $data->name;
        $this->owner = $data->owner;
        $this->state = $data->state;
        $this->mimeType = $data->mime_type;
        $this->size = $data->size;
        $this->meta = $data->meta;
        $this->tags = $data->tags;
        $this->extra = $data->extra;
        $this->extension = $data->extension;
        $this->defaultImage = $data->default_image;
        $this->defaultPreview = $data->default_preview;
        $this->duration = $data->duration;
        $this->folder = new Folder($this->client, $data->folder);
        $this->created = new DateTime($data->created);
        $this->updated = new DateTime($data->updated);
        $this->creator = $data->creator;
    }

    public function setDefaultImage($imageId)
    {
        $request = $this->client->put('file/' . $this->uuid . '/image/' . $imageId);
        $this->setFromData($request->getBody()
            ->getContents());
    }

    public function rename($name)
    {
        $request = $this->client->patch('file/' . $this->uuid, [
            'json' => [
                'name' => $name
            ]
        ]);
        $this->setFromData($request->getBody()
            ->getContents());
    }

    public function addMeta($key, $value)
    {
        $this->setFromData($this->client->post('file/' . $this->uuid . '/meta', [
            'json' => [
                'key' => $key,
                'value' => $value
            ]
        ])
            ->getBody()
            ->getContents());
    }

    public function addTag($tag)
    {
        $this->setFromData($this->client->post('file/' . $this->uuid . '/tag', [
            'json' => [
                'value' => $tag
            ]
        ])
            ->getBody()
            ->getContents());
    }

    public function removeTag($id)
    {
        $this->client->delete('file/' . $this->uuid . '/tag/' . $id);
    }

    public function delete()
    {
        $this->client->delete('file/' . $this->uuid);
    }

    public function getProbe()
    {
        if ($this->probe === false) {
            try {
                $data = $this->client->get('file/' . $this->uuid . '/probe')
                    ->getBody()
                    ->getContents();
                $this->probe = json_decode($data);
            } catch (\Exception $e) {
                $this->probe = null;
            }
        }
        
        return $this->probe;
    }

    public function moveFolder($folder)
    {
        if ($folder instanceof Folder) {
            $folder = $folder->getUuid();
        }
        $this->setFromData($this->client->put('file/' . $this->uuid . '/move/' . $folder)
            ->getBody()
            ->getContents());
    }

    public function share(DateTime $expire = null)
    {
        return json_decode($this->client->get('file/' . $this->uuid . '/share', [
            'query' => [
                'expire' => $expire !== null ? $expire->format('c') : null
            ]
        ])
            ->getBody()
            ->getContents());
    }

    /**
     * 
     * @param LocationInterface $locationSource
     */
    public function export(LocationInterface $locationSource)
    {
        $data = $this->client->post('file/' . $this->uuid . '/export', [
            'json' => $locationSource->toArray()
        ])
            ->getBody()
            ->getContents();
        return json_decode($data);
    }

    public function getJobs()
    {
        $data = $this->client->get('file/' . $this->uuid . '/jobs')
            ->getBody()
            ->getContents();
        return json_decode($data);
    }

    public function getActions()
    {
        $data = $this->client->get('file/' . $this->uuid . '/actions')
            ->getBody()
            ->getContents();
        return json_decode($data);
    }

    public function getUuid()
    {
        return $this->uuid;
    }

    public function getName()
    {
        return $this->name;
    }

    public function getOwner()
    {
        return $this->owner;
    }

    public function getState()
    {
        return $this->state;
    }

    public function getMimeType()
    {
        return $this->mimeType;
    }

    public function getSize()
    {
        return $this->size;
    }

    public function getFolder()
    {
        return $this->folder;
    }

    public function getMeta()
    {
        return $this->meta;
    }

    public function getTags()
    {
        return $this->tags;
    }

    public function getExtra()
    {
        return $this->extra;
    }

    public function getCreated()
    {
        return $this->created;
    }

    public function getCreator()
    {
        return $this->creator;
    }

    public function getExtension()
    {
        return $this->extension;
    }

    public function getDefaultImage()
    {
        return $this->defaultImage;
    }

    public function getDefaultVideo()
    {
        return $this->defaultVideo;
    }

    public function getDuration()
    {
        return $this->duration;
    }
}