<?php

declare(strict_types = 1);

namespace IssetBV\UserBundle\Service;

use Doctrine\Common\Persistence\ObjectManager;
use FOS\UserBundle\Model\User;
use FOS\UserBundle\Util\TokenGenerator;
use IssetBV\UserBundle\Entity\LoginAttempt;
use IssetBV\UserBundle\Entity\LoginIpAddress;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Class IpAccessHandler.
 *
 * @author Bart Malestein <bart@isset.nl>
 */
class IpAccessHandler
{
    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var TokenGenerator
     */
    private $tokenGenerator;

    /**
     * @var MailCenter
     */
    private $mailCenter;

    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * AccessHandler constructor.
     *
     * @param ObjectManager $objectManager
     * @param TokenGenerator $tokenGenerator
     * @param MailCenter $mailCenter
     * @param TranslatorInterface $translator
     */
    public function __construct(ObjectManager $objectManager, TokenGenerator $tokenGenerator, MailCenter $mailCenter, TranslatorInterface $translator)
    {
        $this->objectManager = $objectManager;
        $this->tokenGenerator = $tokenGenerator;
        $this->mailCenter = $mailCenter;
        $this->translator = $translator;
    }

    /**
     * @param User $user
     * @param Request $request
     *
     * @return bool
     */
    public function hasIpAccess(User $user, Request $request)
    {
        $ipAddress = $request->headers->has('X-Forwarded-For') ? $request->headers->get('X-Forwarded-For') : $request->getClientIp();
        $userId = $user->getId();
        $loginIpAddress = $this->objectManager->getRepository(LoginIpAddress::class)->findByUser($userId, $ipAddress);
        if ($loginIpAddress === null) {
            $loginIpAddress = new LoginIpAddress($userId, $ipAddress, $this->tokenGenerator->generateToken());
            $this->objectManager->persist($loginIpAddress);
        }
        if (false === $loginIpAddress->isActive() && true === $loginIpAddress->shouldSendEmail()) {
            $this->mailCenter->sendActivationMail($user, $ipAddress, $loginIpAddress->getActivationToken());
            $loginIpAddress->emailSent();
        }
        $loginAttempt = new LoginAttempt($loginIpAddress, $request->headers->get('user-agent'));
        $this->objectManager->persist($loginAttempt);
        $this->objectManager->flush();

        return $loginIpAddress->isActive();
    }

    /**
     * @return string
     */
    public function getIpValidationEmailMessage()
    {
        return $this->translator->trans('login_ip_validation.ip_validation_email_sent', [], 'IssetBVUserBundle');
    }
}
