<?php
declare(strict_types = 1);

namespace IssetBV\UserBundle\Service\Token;

use Doctrine\DBAL\Connection;
use Symfony\Component\Security\Core\Authentication\RememberMe\PersistentTokenInterface;
use Symfony\Component\Security\Core\Exception\TokenNotFoundException;
use Symfony\Component\Security\Core\Authentication\RememberMe\TokenProviderInterface;
use Symfony\Component\Security\Core\Authentication\RememberMe\PersistentToken;
use Doctrine\DBAL\Types\Type as DoctrineType;

/**
 * Class RememberMeTokenProvider.
 *
 * @author Bart Malestein <bart@isset.nl>
 */
class RememberMeTokenProvider implements TokenProviderInterface
{
    /**
     * Doctrine DBAL database connection
     * F.ex. service id: doctrine.dbal.default_connection.
     *
     * @var Connection
     */
    private $conn;

    /**
     * new DoctrineTokenProvider for the RememberMe authentication service.
     *
     * @param Connection $conn
     */
    public function __construct(Connection $conn)
    {
        $this->conn = $conn;
    }

    /**
     * {@inheritdoc}
     */
    public function deleteToken(string $username)
    {
        $sql = 'DELETE FROM rememberme_token'
            . ' WHERE username = :username';

        $paramValues = array(
            'username' => $username
        );

        $paramTypes = array(
            'username' => \PDO::PARAM_STR
        );

        $this->conn->executeUpdate($sql, $paramValues, $paramTypes);
    }

    /**
     * {@inheritdoc}
     */
    public function loadTokenBySeries($series)
    {
        $sql = 'SELECT class, username, value, lastUsed'
            .' FROM rememberme_token WHERE series=:series';
        $paramValues = array('series' => $series);
        $paramTypes = array('series' => \PDO::PARAM_STR);
        $stmt = $this->conn->executeQuery($sql, $paramValues, $paramTypes);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);

        if ($row) {
            return new PersistentToken($row['class'], $row['username'], $series, $row['value'], new \DateTime($row['lastUsed']));
        }

        throw new TokenNotFoundException('No token found.');
    }

    /**
     * {@inheritdoc}
     */
    public function deleteTokenBySeries($series)
    {
        $sql = 'DELETE FROM rememberme_token WHERE series=:series';
        $paramValues = array('series' => $series);
        $paramTypes = array('series' => \PDO::PARAM_STR);
        $this->conn->executeUpdate($sql, $paramValues, $paramTypes);
    }

    /**
     * {@inheritdoc}
     */
    public function updateToken($series, $tokenValue, \DateTime $lastUsed)
    {
        $sql = 'UPDATE rememberme_token SET value=:value, lastUsed=:lastUsed'
            .' WHERE series=:series';
        $paramValues = array('value' => $tokenValue,
            'lastUsed' => $lastUsed,
            'series' => $series,);
        $paramTypes = array('value' => \PDO::PARAM_STR,
            'lastUsed' => DoctrineType::DATETIME,
            'series' => \PDO::PARAM_STR,);
        $updated = $this->conn->executeUpdate($sql, $paramValues, $paramTypes);
        if ($updated < 1) {
            throw new TokenNotFoundException('No token found.');
        }
    }

    /**
     * {@inheritdoc}
     */
    public function createNewToken(PersistentTokenInterface $token)
    {
        $sql = 'INSERT INTO rememberme_token'
            .' (class, username, series, value, lastUsed)'
            .' VALUES (:class, :username, :series, :value, :lastUsed)';
        $paramValues = array('class' => $token->getClass(),
            'username' => $token->getUsername(),
            'series' => $token->getSeries(),
            'value' => $token->getTokenValue(),
            'lastUsed' => $token->getLastUsed(),);
        $paramTypes = array('class' => \PDO::PARAM_STR,
            'username' => \PDO::PARAM_STR,
            'series' => \PDO::PARAM_STR,
            'value' => \PDO::PARAM_STR,
            'lastUsed' => DoctrineType::DATETIME,);
        $this->conn->executeUpdate($sql, $paramValues, $paramTypes);
    }
}