(function () {
    'use strict';

    angular
        .module('upload')
        .service('UploadService', UploadService);

    UploadService.$inject = ['$http', '$q', '$rootScope'];
    function UploadService($http, $q, $rootScope) {
        var self = this;
        var resumablejs = false;
        var getFilesDeferred = false;

        self.currentFiles = [];
        self.getResumableJs = getResumableJs;
        self.getCurrentFiles = getCurrentFiles;
        self.remove = remove;
        //this.updateFile = updateFile;
        loadCurrentFiles();

        function getResumableJs() {
            if (resumablejs === false) {
                resumablejs = new Resumable({
                    target: window.endpointUrl,
                    chunkSize: 6 * 1024 * 1024,
                    simultaneousUploads: 4,
                    permanentErrors: [400, 404, 409, 415, 500, 501],
                    generateUniqueIdentifier: function (file) {
                        var relativePath = file.webkitRelativePath || file.fileName || file.name; // Some confusion in different versions of Firefox
                        var size = file.size;
                        return (window.userHash + '-' + size + '-' + relativePath.replace(/[^0-9a-zA-Z_-]/img, ''));
                    }
                });
                resumablejs.on('fileSuccess', function (file) {
                    updateFile(file)
                        .then(function (result) {
                            $rootScope.$broadcast('upload.complete', file);
                            loadCurrentFiles();
                        });
                });
                resumablejs.on('fileAdded', function (file, event) {
                    if (true !== window.resumablePaused) {
                        resumablejs.upload();
                    }

                    addFile(file)
                        .then(function (result) {
                            loadCurrentFiles();
                        });
                });
                resumablejs.on('fileError', function (file, message) {
                    loadCurrentFiles();
                });
            }
            return resumablejs;
        }

        function getCurrentFiles() {
            return self.currentFiles;
        }

        function loadCurrentFiles() {
            if (false === getFilesDeferred) {
                getFilesDeferred = $q.defer();
                var setCurrentFiles = $http.get(Routing.generate('isset_bv_upload_api_list'));

                setCurrentFiles
                    .then(function (result) {
                        self.currentFiles = result.data;

                        $rootScope.$broadcast('upload.updated');
                        getFilesDeferred.resolve(result);

                        getFilesDeferred = false;
                    }, function (result) {
                        getFilesDeferred.reject(result);

                        getFilesDeferred = false;
                    });
            }

            return getFilesDeferred.promise;
        }

        function remove(file) {
            var urlRoute = Routing.generate('isset_bv_upload_api_remove', {
                uniqueIdentifier: file.uniqueIdentifier
            });

            $http.get(urlRoute)
                .then(function (result) {
                    loadCurrentFiles();
                });
        }

        function updateFile(file) {
            var urlRoute = Routing.generate('isset_bv_upload_api_update', {
                uniqueIdentifier: file.uniqueIdentifier
            });

            var d = {
                progress: file.progress(),
                state: 'completed',
                url: ''
            };

            return $http({
                url: urlRoute,
                method: 'POST',
                data: $.param(d),
                headers: {'Content-Type': 'application/x-www-form-urlencoded'}
            });
        }

        function addFile(file) {
            var d = {
                filename: file.fileName,
                uniqueIdentifier: file.uniqueIdentifier,
                fileSize: file.size
            };

            return $http({
                url: Routing.generate('isset_bv_upload_api_add'),
                method: 'POST',
                data: $.param(d),
                headers: {'Content-Type': 'application/x-www-form-urlencoded'}
            });
        }
    }
})();