(function () {
    'use strict';

    var resumablePaused = false;

    var uploadController = function ($scope, $q, $filter, uploadService) {
        $scope.uploadFiles = [];
        $scope.currentFiles = [];
        $scope.paused = resumablePaused;
        $scope.uploadrRedirectRoute = uploadrRedirectRoute;

        $scope.count = {
            uploaded: 0,
            uploads: 0
        };

        $scope.redirectFile = function (file) {
            if ($scope.uploadrRedirectRoute) {
                var urlRedirect = Routing.generate('isset_bv_upload_redirect', {upload: file.id});
                document.location = urlRedirect;
            }
        };

        $scope.setPause = function (bool) {
            resumablePaused = bool;
            $scope.paused = bool;
            if (!$scope.$$phase) {
                $scope.$apply();
            }
        };

        $scope.getCurentFiles = function () {
            var fileData = uploadService.getCurrentFiles();
            fileData.then(function (result) {
                $scope.currentFiles = result.data;
                $scope.checkFiles();
            });
        };

        $scope.checkFiles = function () {
            angular.forEach($scope.currentFiles, function (currentFile) {
                if (currentFile.state != 'completed') {
                    currentFile.state = 'new';
                }
                angular.forEach($scope.uploadFiles, function (uploadFile) {
                    if (uploadFile.uniqueIdentifier == currentFile.uniqueIdentifier) {
                        if (currentFile.state == 'completed') {
                            uploadFile.completed = true;
                        } else {
                            currentFile.state = 'found';
                        }
                    }
                });
            });


            if ($scope.currentFiles.length > 0) {
                $scope.count.uploaded = $filter('filter')($scope.currentFiles, {'state': 'completed'});
                $scope.count.uploads = $filter('filter')($scope.currentFiles, {'state': 'new'});
            }
        };

        $scope.addUploadFiles = function (files, r) {
            var urlCalls = [];
            angular.forEach(files, function (file) {
                $scope.uploadFiles.push(file);
                file.pause();

                var uploadData = uploadService.addFile(file);
                uploadData.then(function () {
                    file.pause(false);
                    if (resumablePaused !== true) {
                        r.upload();
                    }
                }, function (result) {
                    if (result.status == 409) { // already uploaded
                        file.finished = true;
                    }
                });
                urlCalls.push(uploadData);
            });

            $q.all(urlCalls).finally(function () {
                $scope.getCurentFiles();
            });

            if (!$scope.$$phase) {
                $scope.$apply();
            }
        };

        $scope.remove = function (file) {
            var uploadData = uploadService.remove(file);
            uploadData.then(function () {
                $scope.getCurentFiles();
            });
        };

        $scope.updateUploadFile = function (file) {
            angular.forEach($scope.uploadFiles, function (uploadFile) {
                if (uploadFile.uniqueIdentifier === file.uniqueIdentifier) {
                    var progress = uploadFile.progress();
                    uploadFile.percent = (progress * 100).toFixed(2);
                    uploadFile.loaded = progress * uploadFile.size;

                    if (!$scope.$$phase) {
                        $scope.$apply();
                    }
                }
            });
        };

        $scope.updateUploadFinish = function (file) {
            angular.forEach($scope.uploadFiles, function (uploadFile) {
                if (uploadFile.uniqueIdentifier === file.uniqueIdentifier) {
                    var remoteData = uploadService.updateFile(file);

                    remoteData.then(function (result) {
                        $scope.getCurentFiles();
                        if (!$scope.$$phase) {
                            $scope.$apply();
                        }
                    });
                }
            });
        };

        $scope.getCurentFiles();
    };

    var uploadService = function ($http) {

        var resumablejs = false;

        this.getResumableJs = function () {
            if (resumablejs === false) {
                resumablejs = new Resumable({
                    target: uploadUrl,
                    chunkSize: 6 * 1024 * 1024,
                    simultaneousUploads: 4,
                    generateUniqueIdentifier: function (file) {
                        var relativePath = file.webkitRelativePath || file.fileName || file.name; // Some confusion in different versions of Firefox
                        var size = file.size;
                        return (window.userHash + '-' + size + '-' + relativePath.replace(/[^0-9a-zA-Z_-]/img, ''));
                    }
                });
            }
            return resumablejs;
        };

        this.getCurrentFiles = function () {
            return $http.get(Routing.generate('isset_bv_upload_api_list'));
        };

        this.remove = function (file) {
            var urlRoute = Routing.generate('isset_bv_upload_api_remove', {
                uniqueIdentifier: file.uniqueIdentifier
            });

            return $http.get(urlRoute);
        };

        this.updateFile = function (file) {
            var urlRoute = Routing.generate('isset_bv_upload_api_update', {
                uniqueIdentifier: file.uniqueIdentifier
            });

            var d = {
                progress: file.progress(),
                state: 'completed',
                url: ''
            };

            return $http({
                url: urlRoute,
                method: 'POST',
                data: $.param(d),
                headers: {'Content-Type': 'application/x-www-form-urlencoded'}
            });
        };

        this.addFile = function (file) {
            var d = {
                filename: file.fileName,
                uniqueIdentifier: file.uniqueIdentifier,
                fileSize: file.size
            };

            return $http({
                url: Routing.generate('isset_bv_upload_api_add'),
                method: 'POST',
                data: $.param(d),
                headers: {'Content-Type': 'application/x-www-form-urlencoded'}
            });
        };
    };

    angular
        .module('uploadController', [])
        .controller('uploadController', [
            '$scope',
            '$q',
            '$filter',
            'uploadService',
            uploadController
        ])
        .service('uploadService', ['$http',
            uploadService
        ])
        .directive('uploadFilesTest', ['uploadService',
            function (uploadService) {
                return {
                    restrict: 'A',
                    scope: '=',
                    link: function (scope, element, attrs) {
                        var r = uploadService.getResumableJs();

                        r.on('fileProgress', function (file) {
                            //console.log(file.fileName, file.progress());
                        });
                    }
                };
            }
        ])
        .directive('uploadFiles', ['uploadService',
            function (uploadService) {
                return {
                    restrict: 'A',
                    scope: '=',
                    link: function (scope, element, attrs) {
                        var r = uploadService.getResumableJs();

                        if (!r.support) {
                            return;
                        } else {

                            if (r.files.length > 0) {
                                scope.addUploadFiles(r.files, r);
                            }

                            var b = $('#drop-target');

                            $(document).on('click', ".resume-button", function () {
                                r.upload();
                                resumablePaused = false;
                            });

                            $(document).on('click', ".pause-button", function () {
                                r.pause();
                                resumablePaused = true;
                            });


                            r.assignBrowse(b);
                            r.assignDrop(b);

                            r.on('fileSuccess', function (file) {
                                scope.updateUploadFinish(file);
                            });
                            r.on('fileProgress', function (file) {
                                scope.updateUploadFile(file);
                            });
                            r.on('fileAdded', function (file, event) {
                                scope.addUploadFiles([file], r);
                            });
                        }
                    }
                };
            }
        ]);
})();