<?php

namespace IssetBV\UploadBundle\Controller;

use Doctrine\Common\Persistence\ObjectManager;
use FOS\RestBundle\Controller\FOSRestController;
use IssetBV\UploadBundle\Entity\Upload;
use IssetBV\UploadBundle\Presenter\UploadPresenter;
use League\Uri\Schemes\Ftp;
use League\Uri\Schemes\Http;
use League\Uri\UriParser;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\ConflictHttpException;

/**
 * Class UploadApiController
 *
 * This controller contains the functionality of a simple API used by the Update module to sync some data back to VT
 *
 * @package IssetBV\UploadBundle\Controller
 * @author Tim Fennis <tim@isset.nl>
 * @author Casper Houde <casper@isset.nl>
 */
class UploadApiController extends FOSRestController
{
    /**
     * @return bool|\DateTime
     */
    private function getTimeoutDate()
    {
        if ($this->container->hasParameter('isset_bv_upload.timeout')) {
            $timeout = $this->container->getParameter('isset_bv_upload.timeout');

            return new \DateTime($timeout);
        }

        return false;
    }

    /**
     * README:
     *
     * For this functionality I had two choices. 1: Let the upload script (endpoint.php) communicate back to the database
     * what the final location of the file is going to be to determine the URL. 2: Improvise some type of convention based
     * one the file's unique identifier to find the file location. This function provides the latter functionality.
     *
     * It translates the data as follows:
     *  unique-identifier: c81e728d9d4c2f636f067f89cc14862c-12999639-MEDIC52Packshot1080png
     *  original filename: MEDIC52-Packshot-1080.png
     *  url: http://koa.dev/uploads/data/c81e728d9d4c2f636f067f89cc14862c-12999639-MEDIC52Packshot1080png.png
     *
     * The cdn.base_url is used to generate the path prefix and for the moment the /data/ sub directory is determined
     * by convention because endpoint.php uses this to store the files.
     *
     *
     *
     * @param $uniqueIdentifier
     * @param $filename
     * @return string
     */
    private function translateUniqueIdentifierAndFilenameToUrl($uniqueIdentifier, $filename)
    {
        $outputFilename = $uniqueIdentifier;

        $extension = '';
        if (strrpos($filename, '.') !== false) {
            $extension = substr($filename, strrpos($filename, '.') + 1); // everything after last dot is extension
        }

        if ($extension !== '') {
            $outputFilename .= '.' . $extension;
        }

        $parser = new UriParser();
        $data = $parser->parse($this->getParameter('isset_bv_upload.public_base_url'));

        if (true === isset($data['scheme'])) {

            if ($data['scheme'] === 'http') {

                $url = Http::createFromString($this->getParameter('isset_bv_upload.public_base_url'));

                return $url->withPath($url->getPath() . $outputFilename)->__toString();
            } elseif ($data['scheme'] === 'ftp') {

                $url = Ftp::createFromString($this->getParameter('isset_bv_upload.public_base_url'));

                return $url->withPath($url->getPath() . $outputFilename)->__toString();
            }
        }

        throw new \RuntimeException('Cannot process ' . $this->getParameter('isset_bv_upload.public_base_url'));

    }

    /**
     * @param Request $request
     * @return Response
     */
    public function addAction(Request $request)
    {
        if (false === $request->request->has('uniqueIdentifier') || false === $request->request->has('filename')) {
            throw new BadRequestHttpException('You must supply a uniqueIdentifier and filename');
        }

        $uniqueIdentifier = strval($request->get('uniqueIdentifier'));

        /** @var ObjectManager $manager */
        $manager = $this->getDoctrine()->getManager();

        /** @var Upload $existingUpload */
        $existingUpload = $manager->getRepository('IssetBVUploadBundle:Upload')->findOneByUniqueIdentifier($uniqueIdentifier);

        if ($existingUpload !== null && $existingUpload->getState() === 'completed') {
            throw new ConflictHttpException('An entity with that identifier already exists');
        } else if ($existingUpload !== null) {
            return new Response('', 200); // Return a 200 to indicate that we can resume this upload
        } else {
            $accountUuid = $this->getUser()->getEntity()->getUuid();
            $originalFilename = strval($request->get('filename'));
            $fileSize = intval($request->get('fileSize'));
            $url = $this->translateUniqueIdentifierAndFilenameToUrl($uniqueIdentifier, $originalFilename);
            $upload = new Upload($uniqueIdentifier, $originalFilename, $url, $fileSize, $accountUuid);

            $manager->persist($upload);
            $manager->flush();

            return new Response('', 201);
        }
    }

    /**
     * @return Response
     */
    public function listAction()
    {
        $accountUuid = $this->getUser()->getEntity()->getUuid();
        $uploads = $this->getDoctrine()->getRepository('IssetBVUploadBundle:Upload')->findByUser($accountUuid, $this->getTimeoutDate());

        return new JsonResponse((new UploadPresenter())->present($uploads), 200);
    }

    /**
     * @param string $uniqueIdentifier
     * @return Response
     */
    public function removeAction($uniqueIdentifier)
    {
        /** @var ObjectManager $manager */
        $manager = $this->getDoctrine()->getManager();
        $accountUuid = $this->getUser()->getEntity()->getUuid();

        /** @var Upload $upload */
        $upload = $manager->getRepository('IssetBVUploadBundle:Upload')->findOneByUniqueIdentifier($uniqueIdentifier);

        if (null === $upload) {
            throw $this->createNotFoundException('Upload not found');
        }

        $upload->delete();
        $manager->flush();

        return new Response('', 201);
    }

    /**
     * @param Request $request
     * @param string $uniqueIdentifier
     * @return Response
     */
    public function updateAction(Request $request, $uniqueIdentifier)
    {
        /** @var ObjectManager $manager */
        $manager = $this->getDoctrine()->getManager();
        $accountUuid = $this->getUser()->getEntity()->getUuid();

        /** @var Upload $upload */
        $upload = $manager->getRepository('IssetBVUploadBundle:Upload')->findOneByUniqueIdentifier($uniqueIdentifier);

        if ($upload->getAccountUuid() !== $accountUuid) {
            throw $this->createAccessDeniedException('You do not have permission to update this resource');
        }

        $upload->setProgress(floatval($request->get('progress')));
        $upload->setState(strval($request->get('state')));

        $manager->flush();

        return new Response('', 200);
    }
}