<?php

namespace IssetBV\UploadBundle\Controller;

use FOS\RestBundle\Controller\FOSRestController;
use IssetBV\UploadBundle\Entity\Upload;
use IssetBV\UploadBundle\Presenter\UploadPresenter;
use IssetBV\UrlBundle\Url;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\ConflictHttpException;

/**
 * Class UploadApiController
 *
 * This controller contains the functionality of a simple API used by the Update module to sync some data back to VT
 *
 * @package IssetBV\UploadBundle\Controller
 * @author Tim Fennis <tim@isset.nl>, Casper Houde <casper@isset.nl>
 */
class UploadApiController extends FOSRestController
{
    /**
     * @param Request $request
     * @return Response
     */
    public function addAction(Request $request)
    {
        if (false === $request->request->has('uniqueIdentifier') || false === $request->request->has('filename')) {
            throw new BadRequestHttpException('You must supply a uniqueIdentifier and filename');
        }

        $uniqueIdentifier = strval($request->get('uniqueIdentifier'));
        $filename = strval($request->get('filename'));

        // @todo sanitize filename

        $manager = $this->getDoctrine()->getManager();
        $accountUuid = $this->getUser()->getEntity()->getUuid();
        $previousUpload = $manager->getRepository('IssetBVUploadBundle:Upload')->findByUniqueIdentifier($uniqueIdentifier, $accountUuid);

//        \Doctrine\Common\Util\Debug::dump($previousUpload->getProgress());
        if ($previousUpload !== null && $previousUpload->getState() === 'completed') {
            throw new ConflictHttpException('An entity with that identifier already exists');
        } else if ($previousUpload !== null) {
            return new Response('', 200); // Return a 200 to indicate that we can resume this upload
        }

        $upload = new Upload(
            $uniqueIdentifier,
            $filename,
            $this->getUser()->getEntity()->getUuid()
        );

        $upload->setFileSize(strval($request->get('fileSize')));

        $manager->persist($upload);
        $manager->flush();

        return new Response('', 201);
    }

    /**
     *
     * @return array
     */
    public function listAction()
    {
        $accountUuid = $this->getUser()->getEntity()->getUuid();
        $uploads = $this->getDoctrine()->getRepository('IssetBVUploadBundle:Upload')->findByUser($accountUuid);

        return new JsonResponse((new UploadPresenter())->present($uploads), 200);
    }

    /**
     * @param mixed $uniqueIdentifier
     * @return Response
     */
    public function removeAction($uniqueIdentifier)
    {
        $em = $this->getDoctrine()->getManager();
        $accountUuid = $this->getUser()->getEntity()->getUuid();
        $upload = $em->getRepository('IssetBVUploadBundle:Upload')->findByUniqueIdentifier($uniqueIdentifier, $accountUuid);

        if(is_null($upload)) {
            throw new AccessDeniedHttpException('You do not have permission to update this resource');
        }

        $upload->delete();
        $em->flush();

        return new Response('', 201);
    }

    /**
     * @param Request $request
     * @param mixed $uniqueIdentifier
     * @return Response
     */
    public function updateAction(Request $request, $uniqueIdentifier)
    {
        $em = $this->getDoctrine()->getManager();
        $accountUuid = $this->getUser()->getEntity()->getUuid();
        $upload = $em->getRepository('IssetBVUploadBundle:Upload')->findByUniqueIdentifier($uniqueIdentifier, $accountUuid);

        if(is_null($upload)) {
            throw new AccessDeniedHttpException('You do not have permission to update this resource');
        }

        $upload->setProgress(floatval($request->get('progress')));
        $upload->setState(strval($request->get('state')));

        if ($request->get('url') !== '') {
            $url = strval($request->get('url'));
        } else {
            $url = $upload->getUrl();
            if($url == "") {
                $url = sprintf("%s%s", $this->container->getParameter('job.builder.upload.folder.ftp'), $upload->getFileName());
            }
        }
        $upload->setUrl($url);

        $em->flush();

        return new Response('', 200);
    }
}