<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Cli;

use InvalidArgumentException;
use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentIssuerRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentMethodRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use IssetBV\TalosBundle\Cli\Renderer\PaymentRenderer;
use League\Tactician\CommandBus;
use Money\Exception\ParserException;
use Money\MoneyParser;
use PhpOption\Option;
use RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\NullOutput;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class CreatePaymentCommand.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class CreatePaymentCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var MoneyParser
     */
    private $moneyParser;

    /**
     * @var OutputInterface
     */
    private $output;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var PaymentIssuerRepository
     */
    private $paymentIssuerRepository;

    /**
     * @var PaymentMethodRepository
     */
    private $paymentMethodRepository;

    /**
     * @var PaymentRenderer
     */
    private $paymentRenderer;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    public function __construct(
        CommandBus $commandBus,
        MoneyParser $moneyParser,
        PaymentIssuerRepository $paymentIssuerRepository,
        PaymentMethodRepository $paymentMethodRepository,
        PaymentRepository $paymentRepository,
        InvoiceFactory $invoiceFactory,
        PaymentFactory $paymentFactory,
        PaymentRenderer $paymentRenderer
    ) {
        parent::__construct('talos:payment:create');

        $this->commandBus = $commandBus;
        $this->moneyParser = $moneyParser;
        $this->paymentIssuerRepository = $paymentIssuerRepository;
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->invoiceFactory = $invoiceFactory;
        $this->paymentFactory = $paymentFactory;
        $this->paymentRepository = $paymentRepository;
        $this->output = new NullOutput();
        $this->paymentRenderer = $paymentRenderer;
    }

    protected function configure()
    {
        $this->addArgument('amount', InputArgument::REQUIRED, ' The amount of money you want to add');
        $this->addArgument('type', InputArgument::REQUIRED, 'OneOff, First or Recurring');
        $this->addArgument('method', InputArgument::REQUIRED, 'The payment method');
        $this->addArgument('issuer', InputArgument::OPTIONAL, 'The issuer code');

        $this->addOption('original-payment-key', 'o', InputOption::VALUE_REQUIRED, 'The paymentkey of the original payment');
        $this->addOption('wallet', 'w', InputOption::VALUE_REQUIRED, 'The remote identifier of the wallet', null);
        $this->addOption('return-url', 'url', InputOption::VALUE_REQUIRED, 'The return URL you want to redirect to after the payment', 'https://www.google.com/');
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @throws InvalidArgumentException
     * @throws ParserException
     *
     * @return int|null|void
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->output = $output;

        $paymentMethod = $this->findMethod($input);
        $issuer = $this->findIssuer($input)->getOrElse(null);

        $paymentType = $this->toPaymentType($input->getArgument('type'));

        if ($paymentType->isRecurring()) {
            $payment = $this->paymentFactory->createRecurringPayment(
                $this->invoiceFactory->createInvoice(),
                $this->moneyParser->parse($input->getArgument('amount'), 'EUR'),
                $this->paymentRepository->findOneByRemoteIdentifier($input->getOption('original-payment-key'))->get()
            );
        } else {
            $payment = $this->paymentFactory->createPayment(
                $this->invoiceFactory->createInvoice(),
                $this->moneyParser->parse($input->getArgument('amount'), 'EUR'),
                $paymentMethod->getOrThrow(new InvalidArgumentException('Cannot find payment method')),
                $issuer,
                $paymentType
            );
        }

        if ($payment instanceof ExecutablePayment) {
            $this->commandBus->handle(new ExecutePaymentCommand($payment->getId(), $input->getOption('return-url')));
        } else {
            throw new RuntimeException('The payment created by the payment factory was not executable');
        }

        $this->paymentRenderer->render(
            $this->paymentRepository->optionallyFind($payment->getId())->getOrThrow(new RuntimeException()),
            $this->output
        );
    }

    private function findMethod(InputInterface $input): Option
    {
        return $this->paymentMethodRepository->findByServiceName($input->getArgument('method'));
    }

    private function findIssuer(InputInterface $input): Option
    {
        return Option::fromValue($input->getArgument('issuer'))
            ->flatMap(function (string $issuerCode) {
                return $this->paymentIssuerRepository->findOneByCode($issuerCode);
            });
    }

    private function toPaymentType(string $paymentTypeString)
    {
        switch (mb_strtolower($paymentTypeString)) {
            case 'first':
                return Payment\PaymentType::createFirst();
            case 'recurring':
                return Payment\PaymentType::createRecurring();
            case 'oneoff':
                return Payment\PaymentType::createOneOff();
            default:
                throw new InvalidArgumentException("Invalid payment type '$paymentTypeString'");
        }
    }
}
