<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Service;

use DateTime;
use DateTimeInterface;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Entity\SubscriptionTerm;
use IssetBV\TalosBundle\Domain\TalosStatus;
use function Functional\every;
use function Functional\some;

class SubscriptionStatusUpdater
{
    /**
     * @param Subscription $subscription
     */
    public function updateSubscription(Subscription $subscription)
    {
        $this->updateSubscriptionFor($subscription, new DateTime());
    }

    /**
     * @param Subscription $subscription
     * @param DateTimeInterface $moment The moment you want to update the status for
     */
    public function updateSubscriptionFor(Subscription $subscription, DateTimeInterface $moment)
    {
        $subscription->updateDenormalizedStatus($this->determineStatus($subscription, $moment));
    }

    private function determineStatus(Subscription $subscription, DateTimeInterface $moment): SubscriptionStatus
    {
        switch (true) {
            case $this->aPaymentIsInvalid($subscription):
                return SubscriptionStatus::invalidated();
            case $this->aPaymentIsPending($subscription):
                return SubscriptionStatus::pending();
            case $this->allTermsExpired($subscription, $moment):
                return SubscriptionStatus::expired();
            case $this->anActiveTermHasASuccessfulEnoughPayment($subscription, $moment):
                return SubscriptionStatus::valid();
            default:
                return SubscriptionStatus::invalidated();
        }
    }

    /**
     * Returns true if a payment linked to this subscription is invalidated.
     *
     * Returns false if this subscription has 0 payments
     *
     * @param Subscription $subscription
     *
     * @return bool
     */
    private function aPaymentIsInvalid(Subscription $subscription)
    {
        return some($subscription->getPayments(), function (Payment $payment) {
            return in_array($payment->getRemoteStatus(), [
                TalosStatus::EXPIRED,
                TalosStatus::CANCELED_BY_BACK_OFFICE,
                TalosStatus::CANCELED_BY_USER,
                TalosStatus::FAILED,
                TalosStatus::REJECTED,
                TalosStatus::TECHNICAL_ERROR,
            ]);
        });
    }

    /**
     * Returns true if a payment linked to this subscription is pending.
     *
     * Returns false if this subscription has 0 payments
     *
     * @param Subscription $subscription
     *
     * @return bool
     */
    private function aPaymentIsPending(Subscription $subscription)
    {
        return some($subscription->getPayments(), function (Payment $payment) {
            return in_array($payment->getRemoteStatus(), [
                TalosStatus::NONE,

                // ref #18919 processing is nog longer treated as pending, it's now treated as successful
                // TalosStatus::PROCESSING,
                TalosStatus::ON_HOLD,
                TalosStatus::PENDING_APPROVAL,
                TalosStatus::AWAITING_USER,
            ]);
        });
    }

    /**
     * Returns true if a subscription as one or more active terms for the given subscription at a given point in time.
     *
     * @param Subscription $subscription
     * @param DateTimeInterface $moment
     *
     * @return bool
     */
    private function anActiveTermHasASuccessfulEnoughPayment(Subscription $subscription, DateTimeInterface $moment)
    {
        return some($subscription->getSubscriptionTerms(), function (SubscriptionTerm $subscriptionTerm) use ($moment) {
            return $subscriptionTerm->isActiveOn($moment) && $subscriptionTerm->isValid();
        });
    }

    /**
     * Returns true if all terms for the given subscription are expired on the given moment.
     *
     * @param Subscription $subscription
     * @param DateTimeInterface $moment
     *
     * @return bool
     */
    private function allTermsExpired(Subscription $subscription, DateTimeInterface $moment)
    {
        return every($subscription->getSubscriptionTerms(), function (SubscriptionTerm $subscriptionTerm) use ($moment) {
            return false === $subscriptionTerm->isActiveOn($moment);
        });
    }
}
