<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus;

use IssetBV\PaymentBundle\Domain\Subscription\Event\SubscriptionStatusChangedEvent;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Service\SubscriptionStatusUpdater;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class DenormalizeSubscriptionStatusHandler
{
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    /**
     * @var SubscriptionStatusUpdater
     */
    private $subscriptionStatusUpdater;

    public function __construct(
        SubscriptionRepository $subscriptionRepository,
        EventDispatcherInterface $eventDispatcher,
        SubscriptionStatusUpdater $subscriptionStatusUpdater
    ) {
        $this->subscriptionRepository = $subscriptionRepository;
        $this->eventDispatcher = $eventDispatcher;
        $this->subscriptionStatusUpdater = $subscriptionStatusUpdater;
    }

    public function handle(DenormalizeSubscriptionStatusCommand $command)
    {
        // Quickly make sure the expiration date is still correct
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());
        $subscription->updateDenormalizedExpirationDate();

        // Status part
        $originalStatus = $subscription->getStatus();
        $this->subscriptionStatusUpdater->updateSubscription($subscription);
        $newStatus = $subscription->getStatus();

        // delta
        if (false === $originalStatus->equals($newStatus)) {
            $this->eventDispatcher->dispatch(
                SubscriptionStatusChangedEvent::name(),
                new SubscriptionStatusChangedEvent($subscription->getId(), $originalStatus, $newStatus)
            );
        }
    }

    private function findSubscription($identifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($identifier)
            ->getOrThrow(CannotDenormalizeSubscriptionStatusException::becauseTheSubscriptionCouldNotBeFound($identifier));
    }
}
