<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Service;

use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Invoice\Invoice;
use IssetBV\PaymentBundle\Domain\Invoice\InvoiceNumber;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\PaymentIssuer;
use IssetBV\PaymentBundle\Domain\PaymentMethod;
use IssetBV\TalosBundle\Entity\TalosPaymentMethod;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\ResponseHandler\Factory\PaymentResponseHandlerFactory;
use Money\Currencies\ISOCurrencies;
use Money\Currency;
use Money\Formatter\DecimalMoneyFormatter;
use Money\Money;
use PhpOption\None;
use PhpOption\Option;
use PhpOption\Some;
use PHPUnit\Framework\TestCase;
use Prophecy\Argument;
use function Functional\first;

class PaymentExecutorTest extends TestCase
{
    const PAYMENT_REFERENCE = 'ref123';
    const INVOICE_NUMBER = 'inv123';
    const PAYMENT_DESCRIPTION = 'payment description';
    const PAYMENT_ISSUER_CODE = 'ABNANL2A';
    const PAYMENT_METHOD_SERVICE_NAME = 'iDEAL';

    public function createMoney($amount = 100, $currencyCode = 'EUR')
    {
        return new Money($amount, new Currency($currencyCode));
    }

    /**
     * Barebone test that doesn't really test anything that can be used to create other tests.
     * @coversDefaultClass
     */
    public function testExecutePayment()
    {
        // This thing is final so we can't mock it. But since it's a tested external dependency I think it's safe to use a real instance
        $formatter = new DecimalMoneyFormatter(new ISOCurrencies());

        /** @var ApiGateway|mixed $gateway */
        $gateway = $this->prophesize(ApiGateway::class);

        /** @var ResponseHandler $responseHandler */
        $responseHandler = $this->prophesize(ResponseHandler::class);

        /** @var PaymentResponseHandlerFactory|mixed $paymentResponseHandlerFactory */
        $paymentResponseHandlerFactory = $this->prophesize(PaymentResponseHandlerFactory::class);
        $paymentResponseHandlerFactory->createResponseHandler()->willReturn($responseHandler);
        $paymentResponseHandlerFactory->createResponseHandlerPayment(Argument::any())->willReturn($responseHandler);

        /** @var InvoiceNumber|mixed $invoiceNumber */
        $invoiceNumber = $this->prophesize(InvoiceNumber::class);
        $invoiceNumber->getAsString()->willReturn(self::INVOICE_NUMBER);

        /** @var Invoice|mixed $invoice */
        $invoice = $this->prophesize(Invoice::class);
        $invoice->getNumber()->willReturn($invoiceNumber);

        /** @var PaymentIssuer|mixed $paymentIssuer */
        $paymentIssuer = $this->prophesize(PaymentIssuer::class);
        $paymentIssuer->getCode()->willReturn(self::PAYMENT_ISSUER_CODE);

        /** @var PaymentMethod|mixed $paymentMethod */
        $paymentMethod = $this->prophesize(TalosPaymentMethod::class);
        $paymentMethod->getServiceName()->willReturn(self::PAYMENT_METHOD_SERVICE_NAME);

        /** @var ExecutablePayment|mixed $payment */
        $payment = $this->prophesize(ExecutablePayment::class);
        $payment->getInvoice()->willReturn($invoice);
        $payment->getReference()->willReturn(self::PAYMENT_REFERENCE);
        $payment->getDescription()->willReturn(Option::fromValue(self::PAYMENT_DESCRIPTION)); // test??
        $payment->getAmount()->willReturn($this->createMoney());
        $payment->getType()->willReturn(Payment\PaymentType::createOneOff());
        $payment->getPaymentIssuer()->willReturn(Option::fromValue($paymentIssuer->reveal()));
        $payment->getPaymentMethod()->willReturn($paymentMethod);
        $payment->getOriginalPayment()->willReturn(None::create());

        $paymentExecutor = new PaymentExecutor(
            $formatter,
            $gateway->reveal(),
            $paymentResponseHandlerFactory->reveal()
        );

        $paymentExecutor->executePayment($payment->reveal(), ClientData::empty());

        // Don't cry about no assertions mr. PHPUNIT
        TestCase::assertInstanceOf(PaymentExecutor::class, $paymentExecutor);
    }

    /**
     * @coversDefaultClass
     */
    public function testThatItSendsARequest()
    {
        // This thing is final so we can't mock it. But since it's a tested external dependency I think it's safe to use a real instance
        $formatter = new DecimalMoneyFormatter(new ISOCurrencies());


        $request = null;
        $hackerino = function ($requests) use (&$request) {
            TestCase::assertInternalType('array', $requests);
            TestCase::assertCount(1, $requests);
            $request = first($requests);
            TestCase::assertInstanceOf(Request::class, $request);
            return true;
        };

        /** @var mixed $gateway */
        $gateway = $this->prophesize(ApiGateway::class);
        $gateway->send(Argument::that($hackerino), Argument::exact('Web'))->shouldBeCalled();

        /** @var ResponseHandler $responseHandler */
        $responseHandler = $this->prophesize(ResponseHandler::class);

        /** @var PaymentResponseHandlerFactory|mixed $paymentResponseHandlerFactory */
        $paymentResponseHandlerFactory = $this->prophesize(PaymentResponseHandlerFactory::class);
        $paymentResponseHandlerFactory->createResponseHandler()->willReturn($responseHandler);
        $paymentResponseHandlerFactory->createResponseHandlerPayment(Argument::any())->willReturn($responseHandler);

        /** @var InvoiceNumber|mixed $invoiceNumber */
        $invoiceNumber = $this->prophesize(InvoiceNumber::class);
        $invoiceNumber->getAsString()->willReturn(self::INVOICE_NUMBER);

        /** @var Invoice|mixed $invoice */
        $invoice = $this->prophesize(Invoice::class);
        $invoice->getNumber()->willReturn($invoiceNumber);

        /** @var PaymentIssuer|mixed $paymentIssuer */
        $paymentIssuer = $this->prophesize(PaymentIssuer::class);
        $paymentIssuer->getCode()->willReturn(self::PAYMENT_ISSUER_CODE);

        /** @var PaymentMethod|mixed $paymentMethod */
        $paymentMethod = $this->prophesize(TalosPaymentMethod::class);
        $paymentMethod->getServiceName()->willReturn(self::PAYMENT_METHOD_SERVICE_NAME);

        /** @var ExecutablePayment|mixed $payment */
        $payment = $this->prophesize(ExecutablePayment::class);
        $payment->getInvoice()->willReturn($invoice);
        $payment->getReference()->willReturn(self::PAYMENT_REFERENCE);
        $payment->getDescription()->willReturn(Some::fromValue(self::PAYMENT_DESCRIPTION)); // test??
        $payment->getAmount()->willReturn($this->createMoney());
        $payment->getType()->willReturn(Payment\PaymentType::createOneOff());
        $payment->getPaymentIssuer()->willReturn(Some::fromValue($paymentIssuer->reveal()));
        $payment->getPaymentMethod()->willReturn($paymentMethod);
        $payment->getOriginalPayment()->willReturn(None::create());

        $paymentExecutor = new PaymentExecutor(
            $formatter,
            $gateway->reveal(),
            $paymentResponseHandlerFactory->reveal()
        );

        $paymentExecutor->executePayment($payment->reveal(), ClientData::empty());
        // Through magic suddenly the $request variable is populated
        $req = json_decode(json_encode($request), true); // more magic

        TestCase::assertArrayHasKey('ClientData', $req);
        TestCase::assertArrayHasKey('UserCulture', $req);
        TestCase::assertArrayHasKey('Time', $req);
        TestCase::assertArrayHasKey('Nonce', $req);
        TestCase::assertArrayHasKey('ReturnUrl', $req);
        TestCase::assertArrayHasKey('Services', $req);


        TestCase::assertEquals(self::PAYMENT_METHOD_SERVICE_NAME, $req['Services'][0]['Name']);
        TestCase::assertEquals('Pay', $req['Services'][0]['Action']);
        TestCase::assertEquals('1', $req['Services'][0]['Version']);
        TestCase::assertEquals('Payment', $req['Services'][0]['Fields'][0]['Name']);

        TestCase::assertEquals('Issuer', $req['Services'][0]['Fields'][1]['Name']);
        TestCase::assertEquals(self::PAYMENT_ISSUER_CODE, $req['Services'][0]['Fields'][1]['Value']);

        $payment = $req['Services'][0]['Fields'][0];

        TestCase::assertEquals('InvoiceNumber', $payment['Fields'][0]['Name']);
        TestCase::assertEquals(self::INVOICE_NUMBER, $payment['Fields'][0]['Value']);

        TestCase::assertEquals('Reference', $payment['Fields'][1]['Name']);
        TestCase::assertEquals(self::PAYMENT_REFERENCE, $payment['Fields'][1]['Value']);

        TestCase::assertEquals('Description', $payment['Fields'][2]['Name']);
        TestCase::assertEquals(self::PAYMENT_DESCRIPTION, $payment['Fields'][2]['Value']);

        TestCase::assertEquals('Currency', $payment['Fields'][3]['Name']);
        TestCase::assertEquals('EUR', $payment['Fields'][3]['Value']);

        TestCase::assertEquals('RecurrentType', $payment['Fields'][4]['Name']);
        TestCase::assertEquals('OneOff', $payment['Fields'][4]['Value']);

        TestCase::assertEquals('AmountDebit', $payment['Fields'][5]['Name']);
        TestCase::assertEquals('1.00', $payment['Fields'][5]['Value']);
    }

    /**
     * @coversDefaultClass
     */
    public function testThatItSendsARecurringRequest()
    {
        // This thing is final so we can't mock it. But since it's a tested external dependency I think it's safe to use a real instance
        $formatter = new DecimalMoneyFormatter(new ISOCurrencies());


        $request = null;
        $hackerino = function ($requests) use (&$request) {
            TestCase::assertInternalType('array', $requests);
            TestCase::assertCount(1, $requests);
            $request = first($requests);
            TestCase::assertInstanceOf(Request::class, $request);
            return true;
        };

        /** @var mixed $gateway */
        $gateway = $this->prophesize(ApiGateway::class);
        $gateway->send(Argument::that($hackerino), Argument::exact('BatchFile'))->shouldBeCalled();

        /** @var ResponseHandler $responseHandler */
        $responseHandler = $this->prophesize(ResponseHandler::class);

        /** @var PaymentResponseHandlerFactory|mixed $paymentResponseHandlerFactory */
        $paymentResponseHandlerFactory = $this->prophesize(PaymentResponseHandlerFactory::class);
        $paymentResponseHandlerFactory->createResponseHandler()->willReturn($responseHandler);
        $paymentResponseHandlerFactory->createResponseHandlerPayment(Argument::any())->willReturn($responseHandler);

        /** @var InvoiceNumber|mixed $invoiceNumber */
        $invoiceNumber = $this->prophesize(InvoiceNumber::class);
        $invoiceNumber->getAsString()->willReturn(self::INVOICE_NUMBER);

        /** @var Invoice|mixed $invoice */
        $invoice = $this->prophesize(Invoice::class);
        $invoice->getNumber()->willReturn($invoiceNumber);

        /** @var PaymentIssuer|mixed $paymentIssuer */
        $paymentIssuer = $this->prophesize(PaymentIssuer::class);
        $paymentIssuer->getCode()->willReturn(self::PAYMENT_ISSUER_CODE);

        /** @var PaymentMethod|mixed $paymentMethod */
        $paymentMethod = $this->prophesize(TalosPaymentMethod::class);
        $paymentMethod->getServiceName()->willReturn(self::PAYMENT_METHOD_SERVICE_NAME);

        /** @var ExecutablePayment|mixed $originalPayment */
        $originalPayment = $this->prophesize(ExecutablePayment::class);
        $originalPayment->getRemoteIdentifier()->willReturn(Some::fromValue('AB12'));

        /** @var ExecutablePayment|mixed $payment */
        $payment = $this->prophesize(ExecutablePayment::class);
        $payment->getInvoice()->willReturn($invoice);
        $payment->getReference()->willReturn(self::PAYMENT_REFERENCE);
        $payment->getDescription()->willReturn(Some::fromValue(self::PAYMENT_DESCRIPTION)); // test??
        $payment->getAmount()->willReturn($this->createMoney());
        $payment->getType()->willReturn(Payment\PaymentType::createOneOff());
        $payment->getPaymentIssuer()->willReturn(Some::fromValue($paymentIssuer->reveal()));
        $payment->getPaymentMethod()->willReturn($paymentMethod);
        $payment->getOriginalPayment()->willReturn(Some::fromValue($originalPayment->reveal()));

        $paymentExecutor = new PaymentExecutor(
            $formatter,
            $gateway->reveal(),
            $paymentResponseHandlerFactory->reveal()
        );

        $paymentExecutor->executePayment($payment->reveal(), ClientData::empty());
        // Through magic suddenly the $request variable is populated
        $req = json_decode(json_encode($request), true); // more magic

        TestCase::assertArrayHasKey('ClientData', $req);
        TestCase::assertArrayHasKey('UserCulture', $req);
        TestCase::assertArrayHasKey('Time', $req);
        TestCase::assertArrayHasKey('Nonce', $req);
        TestCase::assertArrayHasKey('ReturnUrl', $req);
        TestCase::assertArrayHasKey('Services', $req);


        TestCase::assertEquals(self::PAYMENT_METHOD_SERVICE_NAME, $req['Services'][0]['Name']);
        TestCase::assertEquals('PayRecurrent', $req['Services'][0]['Action']);
        TestCase::assertEquals('1', $req['Services'][0]['Version']);
        TestCase::assertEquals('Payment', $req['Services'][0]['Fields'][0]['Name']);

        TestCase::assertEquals('Issuer', $req['Services'][0]['Fields'][1]['Name']);
        TestCase::assertEquals(self::PAYMENT_ISSUER_CODE, $req['Services'][0]['Fields'][1]['Value']);

        $payment = $req['Services'][0]['Fields'][0];

        TestCase::assertEquals('InvoiceNumber', $payment['Fields'][0]['Name']);
        TestCase::assertEquals(self::INVOICE_NUMBER, $payment['Fields'][0]['Value']);

        TestCase::assertEquals('Reference', $payment['Fields'][1]['Name']);
        TestCase::assertEquals(self::PAYMENT_REFERENCE, $payment['Fields'][1]['Value']);

        TestCase::assertEquals('Description', $payment['Fields'][2]['Name']);
        TestCase::assertEquals(self::PAYMENT_DESCRIPTION, $payment['Fields'][2]['Value']);

        TestCase::assertEquals('Currency', $payment['Fields'][3]['Name']);
        TestCase::assertEquals('EUR', $payment['Fields'][3]['Value']);

        TestCase::assertEquals('RecurrentType', $payment['Fields'][4]['Name']);
        TestCase::assertEquals('OneOff', $payment['Fields'][4]['Value']);

        TestCase::assertEquals('AmountDebit', $payment['Fields'][5]['Name']);
        TestCase::assertEquals('1.00', $payment['Fields'][5]['Value']);
    }
}
