<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\PaymentIssuer;
use IssetBV\PaymentBundle\Domain\PaymentMethod;
use IssetBV\TalosBundle\Storage\CreatedUpdatedFields;

/**
 * Class TalosIssuer.
 *
 * @author Tim Fennis <tim@isset.nl>
 * @ORM\Entity(repositoryClass="IssetBV\TalosBundle\Repository\DoctrinePaymentIssuerRepository")
 * @ORM\Table(name="talos__payment_issuers")
 * @ORM\HasLifecycleCallbacks()
 */
class TalosPaymentIssuer implements PaymentIssuer
{
    use CreatedUpdatedFields;
    /**
     * @var int
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var PaymentMethod
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\PaymentMethod", inversedBy="issuers")
     * @ORM\JoinColumn(name="payment_method_id", referencedColumnName="id")
     */
    private $paymentMethod;

    /**
     * @var bool
     * @ORM\Column(name="seen_in_last_update", type="boolean", nullable=false)
     */
    private $seenInLastUpdate;

    /**
     * @var string
     * @ORM\Column(name="value", type="string", nullable=false, unique=true)
     */
    private $value;

    /**
     * @var string
     * @ORM\Column(name="name", type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     * @ORM\Column(name="group_name", type="string", nullable=false)
     */
    private $groupName;

    /**
     * @var bool
     * @ORM\Column(name="is_default", type="boolean", nullable=false)
     */
    private $default;

    /**
     * TalosIssuer constructor.
     *
     * @param string $value
     * @param string $name
     * @param string $groupName
     * @param bool $default
     * @param PaymentMethod $paymentMethod
     */
    public function __construct(string $value, string $name, string $groupName, bool $default, PaymentMethod $paymentMethod)
    {
        $this->initCrudFields();
        $this->value = $value;
        $this->name = $name;
        $this->groupName = $groupName;
        $this->default = $default;
        $this->seenInLastUpdate = true;
        $this->paymentMethod = $paymentMethod;
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getValue(): string
    {
        return $this->value;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getGroupName(): string
    {
        return $this->groupName;
    }

    /**
     * @return bool
     */
    public function isDefault(): bool
    {
        return $this->default;
    }

    /**
     * @param bool $seenInLastUpdate
     */
    public function setSeenInLastUpdate(bool $seenInLastUpdate)
    {
        $this->seenInLastUpdate = $seenInLastUpdate;
    }

    /**
     * @return bool
     */
    public function wasSeenInLastUpdate(): bool
    {
        return $this->seenInLastUpdate;
    }

    /**
     * @param string $name
     * @param string $groupName
     * @param bool $default
     */
    public function update(string $name, string $groupName, bool $default)
    {
        $this->name = $name;
        $this->groupName = $groupName;
        $this->default = $default;
        $this->seenInLastUpdate = true;
    }

    /**
     * @return string
     */
    public function getDisplayName(): string
    {
        return $this->getName();
    }

    /**
     * Returns the code of this issuer to be used in the API.
     *
     * Example: ABNANL2A
     *
     * @see PaymentIssuer
     *
     * @return string
     */
    public function getCode(): string
    {
        return $this->getValue();
    }

    /**
     * @return PaymentMethod
     */
    public function getPaymentMethod(): PaymentMethod
    {
        return $this->paymentMethod;
    }
}
