<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Entity;

use DateInterval;
use DateTime;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\Exception\InvalidPaymentIntervalException;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use IssetBV\TalosBundle\Domain\TalosStatus;
use PhpOption\Option;

/**
 * @author Tim Fennis <tim@isset.nl>
 * @ORM\Entity()
 * @ORM\Table(name="subscriptions_terms")
 */
class SubscriptionTerm
{
    /**
     * @var int
     * @ORM\Id()
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var Subscription
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Subscription\Subscription", inversedBy="subscriptionTerms")
     */
    private $subscription;

    /**
     * @var Payment|null
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment")
     */
    private $payment;

    /**
     * @var DateTime
     * @ORM\Column(name="date_time", type="datetime", nullable=false)
     */
    private $dateTime;

    /**
     * @var Interval
     * @ORM\Embedded(class="Interval")
     */
    private $termInterval;

    /**
     * @var bool
     * @ORM\Column(name="activity_override", type="boolean", nullable=false)
     */
    private $override;

    public function __construct(Subscription $subscription, DateTime $startTime, Payment $payment = null, Interval $termInterval, bool $activityOverride = false)
    {
        $this->subscription = $subscription;
        $this->payment = $payment;
        $this->dateTime = $startTime;
        $this->termInterval = $termInterval;
        $this->override = $activityOverride;
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }



    public function hasSuccessfulPayment(): bool
    {
        return $this->getPayment()
            ->map(function (Payment $payment) {
                // ref #18919 we now also return true if the payment is processing
                return TalosStatus::SUCCESS === $payment->getRemoteStatus() ||
                    TalosStatus::PROCESSING === $payment->getRemoteStatus();
            })
            ->getOrElse(false);
    }

    /**
     * Return true if this term has a successful payment or has override enabled.
     *
     * @return bool
     */
    public function isValid(): bool
    {
        return $this->hasSuccessfulPayment() || $this->isOverrideEnabled();
    }

    /**
     * @return DateInterval
     */
    public function getTermInterval(): DateInterval
    {
        return $this->termInterval->getDateInterval()->getOrThrow(new InvalidPaymentIntervalException());
    }

    /**
     * @return DateTime
     */
    public function getDateTime(): DateTime
    {
        return $this->dateTime;
    }

    /**
     * @return Subscription
     */
    public function getSubscription(): Subscription
    {
        return $this->subscription;
    }

    /**
     * @return Option
     */
    public function getPayment(): Option
    {
        return Option::fromValue($this->payment);
    }

    /**
     * @return bool
     */
    public function isOverrideEnabled(): bool
    {
        return $this->override;
    }

    /**
     * @deprecated use isActiveOn and always explicitly mention the date
     *
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->isActiveOn(new DateTime());
    }

    /**
     * @param DateTimeInterface $targetDateTime
     *
     * @return bool
     */
    public function isActiveOn(DateTimeInterface $targetDateTime): bool
    {
        $endDate = self::calculateEndDate($this->getDateTime(), $this->getTermInterval());

        return $targetDateTime->getTimestamp() < $endDate->getTimestamp();
    }

    /**
     * @return DateTime
     */
    public function getEndDate(): DateTime
    {
        // calc the end date
        $dateTime = self::calculateEndDate($this->dateTime, $this->getTermInterval());

        // convert to mutable for compatibility
        return new DateTime($dateTime->format('c'));
    }

    /**
     * @param DateTime $startDate
     * @param DateInterval $interval
     *
     * @return DateTimeImmutable
     */
    private static function calculateEndDate(DateTime $startDate, DateInterval $interval): DateTimeImmutable
    {
        return DateTimeImmutable::createFromMutable($startDate)->add($interval);
    }
}
