<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\RenewSubscription;

use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\Subscription\RenewAmountProvider;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use League\Tactician\CommandBus;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionHandler
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var RenewAmountProvider
     */
    private $renewAmountProvider;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(
        CommandBus $commandBus,
        InvoiceFactory $invoiceFactory,
        PaymentFactory $paymentFactory,
        SubscriptionRepository $subscriptionRepository,
        RenewAmountProvider $renewAmountProvider
    ) {
        $this->commandBus = $commandBus;
        $this->invoiceFactory = $invoiceFactory;
        $this->paymentFactory = $paymentFactory;
        $this->subscriptionRepository = $subscriptionRepository;
        $this->renewAmountProvider = $renewAmountProvider;
    }

    public function handle(RenewSubscriptionCommand $command)
    {
        // First update the subscription status (probably again since we should have done that already)
        $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($command->getSubscriptionIdentifier()));

        // Load the subscription from the database
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());

        // Make sure this subscription should be renewed
        $this->guardAgainstRenewingNonExpiredSubscriptions($subscription);
        $this->guardAgainstRenewingCanceledSubscriptions($subscription);

        // Find a payment that is renewable (Executable and First)
        $renewablePayment = $subscription->findRenewablePayment()
            ->getOrThrow(CannotRenewSubscriptionException::becauseTheSubscriptionDoesNotHaveAValidPayment($subscription->getId()));

        // Create a new payment/invoice based on the previous payment
        $newPayment = $this->paymentFactory->createRecurringPayment(
            $this->invoiceFactory->createInvoice(),
            $this->renewAmountProvider->findRenewAmount($subscription),
            $renewablePayment
        );

        // Execute the payment
        $this->commandBus->handle(new ExecutePaymentCommand($newPayment->getId()));

        // Link the payment to this subscription
        $subscription->createNewTerm($newPayment); //@todo possibly move this up and don't execute here?

        // Denormalize the subscription again te ensure the state is updated
        $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($command->getSubscriptionIdentifier()));
    }

    private function findSubscription(SubscriptionIdentifier $identifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($identifier)
            ->getOrThrow(CannotRenewSubscriptionException::becauseTheSubscriptionCouldNotBeFound($identifier));
    }

    private function guardAgainstRenewingCanceledSubscriptions(Subscription $subscription)
    {
        if ($subscription->isCanceled()) {
            throw CannotRenewSubscriptionException::becauseTheSubscriptionIsCanceled($subscription->getId());
        }
    }

    private function guardAgainstRenewingNonExpiredSubscriptions(Subscription $subscription)
    {
        if (true !== $subscription->getStatus()->isExpired()) {
            throw CannotRenewSubscriptionException::becauseTheSubscriptionIsNotExpired($subscription->getId());
        }
    }
}
