<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Service;

use DateTime;
use DateTimeInterface;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Identifier\UuidPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Invoice\Invoice;
use IssetBV\PaymentBundle\Domain\Invoice\InvoiceNumber;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Payment\PaymentType;
use IssetBV\PaymentBundle\Domain\PaymentMethod;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Interval;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\TalosBundle\Domain\TalosStatus;
use Money\Currency;
use Money\Money;
use PhpOption\None;
use PhpOption\Option;
use PHPUnit\Framework\TestCase;
use Ramsey\Uuid\Uuid;

class SubscriptionStatusUpdaterFunctionalTest extends TestCase
{
    /**
     * @dataProvider intervalAndStartDateProvider
     *
     * @param Interval $interval
     * @param DateTime $startDate
     * @param Payment|null $payment
     * @param Interval $initialInterval
     * @param DateTime $currentDate
     * @param SubscriptionStatus $expectedStatus
     */
    public function testNewSubscriptions(Interval $interval, DateTime $startDate, Payment $payment = null, Interval $initialInterval = null, DateTime $currentDate, SubscriptionStatus $expectedStatus)
    {
        $subscription = Subscription::createSubscription($interval, $startDate, $payment, $initialInterval);

        $this->doTestSubscription($subscription, $currentDate, $expectedStatus);
    }

    /**
     * @dataProvider expiredSubscriptionDataProvider
     *
     * @param Interval $interval
     * @param DateTime $startDate
     * @param Payment|null $payment
     * @param Interval|null $initialInterval
     * @param DateTime $currentDate
     */
    public function testThatSubscriptionsExpire(Interval $interval, DateTime $startDate, Payment $payment = null, Interval $initialInterval = null, DateTime $currentDate)
    {
        $subscription = Subscription::createSubscription($interval, $startDate, $payment, $initialInterval);

        $this->doTestSubscription($subscription, $currentDate, SubscriptionStatus::expired());
    }

    /**
     * ref #18919 Processing status changes
     *
     * We made a change that causes Processing payments to count as valid, but other pending-like payments to still cause
     * the subscription status to become pending. This test has been changed and now tests if payments stay valid when
     * processing payments are added.
     */
    public function testThatSubscriptionDoesNotBecomePending()
    {
        $subscription = Subscription::createSubscription(Interval::createFromString('1 month'), new DateTime('2017-01-01T12:00:00+00:00'));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::PROCESSING));

        $this->doTestSubscription($subscription, new DateTime('2017-07-15T12:00:00+00:00'), SubscriptionStatus::valid());
        TestCase::assertEquals($subscription->getDateValidUntil()->get(), new DateTime('2017-08-01T12:00:00+00:00'));
    }

    public function testThatASubscriptionBecomesPending()
    {
        $subscription = Subscription::createSubscription(Interval::createFromString('1 month'), new DateTime('2017-01-01T12:00:00+00:00'));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::SUCCESS));
        $subscription->createNewTerm($this->createMockPayment(PaymentType::createOneOff(), TalosStatus::ON_HOLD));

        $this->doTestSubscription($subscription, new DateTime('2017-07-15T12:00:00+00:00'), SubscriptionStatus::pending());
        TestCase::assertEquals($subscription->getDateValidUntil()->get(), new DateTime('2017-08-01T12:00:00+00:00'));
    }

    private function doTestSubscription(Subscription $subscription, DateTime $currentDate, SubscriptionStatus $expectedStatus)
    {
        $subscriptionStatusUpdater = new SubscriptionStatusUpdater();
        $subscriptionStatusUpdater->updateSubscriptionFor($subscription, $currentDate);

        TestCase::assertEquals(
            $expectedStatus->getCode(),
            $subscription->getStatus()->getCode(),
            'Failed asserting that subscription status matches "' . $expectedStatus . '" got "' . $subscription->getStatus() . '"'
        );
    }

    public function expiredSubscriptionDataProvider()
    {
        return [
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::SUCCESS),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-05-10T12:00:00+00:00'), // expired
            ],
        ];
    }

    public function intervalAndStartDateProvider()
    {
        return [
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */null,
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::valid(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::NONE),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::pending(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::SUCCESS),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::valid(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::PROCESSING),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::valid(), // We now expect processing payments to be valid (ref #18919)
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::ON_HOLD),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::pending(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::PENDING_APPROVAL),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::pending(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::AWAITING_USER),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::pending(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::FAILED),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::invalidated(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::REJECTED),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::invalidated(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::CANCELED_BY_USER),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::invalidated(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::CANCELED_BY_BACK_OFFICE),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::invalidated(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::TECHNICAL_ERROR),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::invalidated(),
            ],
            [
                /* subscription interval*/ Interval::createFromString('1 month'),
                /* start date */new DateTime('2017-01-05T12:00:00+00:00'),
                /* initial payment */ $this->createMockPayment(PaymentType::createFirst(), TalosStatus::EXPIRED),
                /* initial interval*/ null,
                /* current date */new DateTime('2017-01-10T12:00:00+00:00'), // valid
                /* expected status */ SubscriptionStatus::invalidated(),
            ],
        ];
    }

    private function createMockPayment(PaymentType $paymentType, string $remoteStatus)
    {
        return new class($paymentType, $remoteStatus) implements Payment {

            private $remoteStatus;
            private $paymentType;

            public function __construct($paymentType, $remoteStatus)
            {
                $this->paymentType = $paymentType;
                $this->remoteStatus = $remoteStatus;
            }

            public function getId(): LocalPaymentIdentifier
            {
                return new UuidPaymentIdentifier(Uuid::uuid4());
            }

            public function getAmount(): Money
            {
                return new Money(299, new Currency('EUR'));
            }

            public function getInvoice(): Invoice
            {
                return new class implements Invoice {

                    /**
                     * @return InvoiceNumber
                     */
                    public function getNumber(): InvoiceNumber
                    {
                        return new class implements InvoiceNumber {

                            /**
                             * @return string
                             */
                            public function getAsString(): string
                            {
                                return 'i001';
                            }
                        };
                    }
                };
            }

            /**
             * @return string
             */
            public function getReference(): string
            {
                return 'pay-ref-01';
            }

            /**
             * @return PaymentMethod
             */
            public function getPaymentMethod(): PaymentMethod
            {
                return new class implements PaymentMethod {

                    public function getId(): int
                    {
                        return 1;
                    }

                    public function getDisplayName(): string
                    {
                        return 'TestMethod';
                    }


                    public function getIssuers()
                    {
                        return [];
                    }


                    public function shouldHaveIssuers(): bool
                    {
                        return false;
                    }

                    public function getWallet(): Option
                    {
                        return None::create();
                    }

                    /**
                     * Returns true if this payment method is currently active.
                     *
                     * @return bool
                     */
                    public function isActive(): bool
                    {
                        return true;
                    }

                    /**
                     * Returns the date when this payment method becomes active.
                     *
                     * @return DateTimeInterface
                     */
                    public function getDateActive(): DateTimeInterface
                    {
                        return new DateTime('2010-01-01T12:00:00+00:00');
                    }
                };
            }

            public function getPaymentIssuer(): Option
            {
                return None::create();
            }

            public function getRecurrentType(): string
            {
                return (string) $this->getType();
            }

            public function getType(): PaymentType
            {
                return $this->paymentType;
            }

            public function isUsableForAutoTopup(): bool
            {
                return false;
            }


            public function getRemoteIdentifier(): Option
            {
                // random good?
                return strtoupper(Uuid::uuid4()->getHex());
            }

            public function setRemoteIdentifier(string $remoteIdentifier)
            {
                // TODO: Implement setRemoteIdentifier() method.
            }

            public function setRemoteStatus(string $remoteStatusCode)
            {
                // TODO: Implement setRemoteStatus() method.
            }

            public function getRemoteStatus()
            {
                return $this->remoteStatus;
            }
        };
    }
}
