<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\ConfigureAutoRenew;

use IssetBV\PaymentBundle\CommandBus\ConfigureAutoRenew\ConfigureAutoRenewCommand;
use IssetBV\PaymentBundle\CommandBus\CreateAutoTopupProfile\CreateAutoTopupProfileCommand;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\AutoTopupProfileRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Domain\Repository\WalletRepository;
use IssetBV\TalosBundle\Entity\TalosAutoTopupProfile;
use IssetBV\TalosBundle\Entity\TalosWallet;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\GroupField;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\StatusResponseHandler;
use League\Tactician\CommandBus;
use LogicException;
use Money\Money;
use PhpOption\LazyOption;

/**
 * @author Tim Fennis <tim@isset.nl>
 *
 * @todo fix the exception in this class
 */
class ConfigureAutoRenewHandler
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var AutoTopupProfileRepository
     */
    private $autoTopupProfileRepository;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var WalletRepository
     */
    private $walletRepository;

    /**
     * TalosConfigureAutoRenewHandler constructor.
     *
     * @param CommandBus $commandBus
     * @param ApiGateway $gateway
     * @param AutoTopupProfileRepository $autoTopupProfileRepository
     * @param PaymentRepository $paymentRepository
     * @param WalletRepository $walletRepository
     */
    public function __construct(
        CommandBus $commandBus,
        ApiGateway $gateway,
        AutoTopupProfileRepository $autoTopupProfileRepository,
        PaymentRepository $paymentRepository,
        WalletRepository $walletRepository
    ) {
        $this->commandBus = $commandBus;
        $this->gateway = $gateway;
        $this->autoTopupProfileRepository = $autoTopupProfileRepository;
        $this->paymentRepository = $paymentRepository;
        $this->walletRepository = $walletRepository;
    }

    public function handle(ConfigureAutoRenewCommand $command)
    {
        $autoTopupProfile = $this->obtainAutoTopupProfile($command->getActivationBalance(), $command->getCooldown());

        $wallet = $this->walletRepository
            ->findByRemoteIdentifier($command->getWalletIdentifier())
            ->getOrThrow(new LogicException());

        $autoTopupProfileKey = $autoTopupProfile
            ->getRemoteIdentifier()
            ->getOrThrow(new LogicException());

        $payment = $this->findPayment($command);

        $topupSourceWalletKey = $payment
            ->getPaymentMethod()
            ->getWallet()
            ->get()
            ->getIdentifier();

        $request = new Request($command->getClientData()->getOrElse(ClientData::empty()), [
            new Service('WalletSystem', 'Link', [
                new GroupField('AutoTopupLinkInfo', [
                    new SingleField('WalletKey', $command->getWalletIdentifier()),
                    new SingleField('AutoTopupProfileKey', $autoTopupProfileKey),
                    new SingleField('PaymentKey', $payment->getRemoteIdentifier()->getOrThrow(new LogicException())),
                    new SingleField('TopupSourceWalletKey', $topupSourceWalletKey),
                    new SingleField('LinkAction', 'Link'),
                ]),
            ]),
        ]);

        $request->addResponseHandler(new StatusResponseHandler(
            function () use ($wallet, $autoTopupProfile) {
                if ($wallet instanceof TalosWallet) {
                    $wallet->setAutoTopupProfile($autoTopupProfile);
                }
            }
        ));

        $this->gateway->send([$request]);
    }

    private function obtainAutoTopupProfile(Money $activationBalance, int $cooldown): TalosAutoTopupProfile
    {
        return $this->autoTopupProfileRepository
            ->findOneByCooldownAndActivationBalance($cooldown, $activationBalance)
            ->orElse(new LazyOption(function () use ($cooldown, $activationBalance) {
                $this->createAutoTopupProfile($cooldown, $activationBalance);

                return $this->autoTopupProfileRepository->findOneByCooldownAndActivationBalance($cooldown, $activationBalance);
            }))
            ->getOrThrow(new LogicException('Could not find or create an auto topup profile that matches the required description'));
    }

    private function createAutoTopupProfile(int $cooldown, Money $activationBalance)
    {
        $this->commandBus->handle(new CreateAutoTopupProfileCommand($cooldown, null, null, $activationBalance));
    }

    /**
     * @param ConfigureAutoRenewCommand $command
     *
     * @return Payment
     */
    private function findPayment(ConfigureAutoRenewCommand $command): Payment
    {
        $payment = $this->paymentRepository
            ->optionallyFind($command->getPaymentIdentifier())
            ->getOrThrow(new LogicException());

        return $payment;
    }
}
