<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use Doctrine\Common\Collections\Criteria;
use Exception;
use IssetBV\PaymentBundle\Cli\Renderer\SubscriptionRenderer;
use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\CannotRenewSubscriptionException;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class DenormalizeSubscriptionsCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SymfonyStyle
     */
    private $console;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    /**
     * @var SubscriptionRenderer
     */
    private $subscriptionRenderer;

    public function __construct(
        CommandBus $commandBus,
        SubscriptionRepository $subscriptionRepository,
        SubscriptionRenderer $subscriptionRenderer
    ) {
        parent::__construct('subscription:denormalize');
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
        $this->subscriptionRenderer = $subscriptionRenderer;
    }

    public function configure()
    {
        // do nothing for now
        $this->addOption('migration-mode', 'm', InputOption::VALUE_NONE, 'Run this command in migration mode also updating all outdated subscriptions');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->console = new SymfonyStyle($input, $output);
        $this->console->title('Denormalizing subscriptions');

        if ($input->getOption('migration-mode')) {
            $criteria = Criteria::create();
            $criteria->where(Criteria::expr()->isNull('denormalizedExpirationDate'));

            $this->subscriptionRepository->matching($criteria);
            $this->console->note('Running in migration mode');
            $this->console->note('Searching and updating subscriptions without denormalized expirationDate...');

            /** @var Subscription[] $subscriptions */
            $subscriptions = $this->subscriptionRepository->matching($criteria);
            $this->updateSubscriptions($subscriptions);
        }

        $this->console->text('Searching and updating pending subscriptions...');
        $this->updateSubscriptions($this->subscriptionRepository->findPendingSubscriptions());

        $this->console->text('Searching and updating expired subscriptions...');
        $this->updateSubscriptions($this->subscriptionRepository->findExpiredAndValidSubscriptions());
    }

    /**
     * @param Subscription[] $subscriptions
     */
    private function updateSubscriptions($subscriptions)
    {
        $numberOfSubscriptions = count($subscriptions);

        if (0 === $numberOfSubscriptions) {
            $this->console->success('Nothing to update');

            return;
        }

        $this->console->progressStart($numberOfSubscriptions);

        try {
            foreach ($subscriptions as $subscription) {
                try {
                    $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($subscription->getId()));
                } catch (CannotRenewSubscriptionException $e) {
                    $this->console->warning($e->getMessage());
                }

                $this->console->progressAdvance(1);
            }

            $this->console->progressFinish();
            $this->console->success('Subscriptions updated');
        } catch (Exception $e) {
            $this->console->progressFinish();
            $this->console->caution('Exception: ' . $e->getMessage());
        }
    }
}
