<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\Mutation;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\RemoteObject;
use IssetBV\PaymentBundle\Domain\Wallet;
use IssetBV\PaymentBundle\Domain\WalletOwner;
use IssetBV\PaymentBundle\Domain\WalletType;
use IssetBV\TalosBundle\Domain\AutoRenewConfigurationWrapper;
use IssetBV\TalosBundle\Storage\CreatedUpdatedFields;
use Money\Currency;
use Money\Money;
use PhpOption\None;
use PhpOption\Option;

/**
 * Class TalosWallet.
 *
 * @author Tim Fennis <tim@isset.nl>
 * @ORM\Entity(repositoryClass="IssetBV\TalosBundle\Repository\DoctrineWalletRepository")
 * @ORM\Table(name="talos__wallets")
 * @ORM\HasLifecycleCallbacks()
 */
class TalosWallet implements Wallet, RemoteObject
{
    use CreatedUpdatedFields;

    /**
     * @var int
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var Payment|null
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment")
     */
    private $autoTopupPayment;

    /**
     * @var TalosAutoTopupProfile|null
     * @ORM\ManyToOne(targetEntity="IssetBV\TalosBundle\Entity\TalosAutoTopupProfile")
     * @ORM\JoinColumn(name="auto_topup_profile_id", referencedColumnName="id")
     */
    private $autoTopupProfile;

    /**
     * @var bool
     * @ORM\Column(name="has_sub_wallets", type="boolean", nullable=false)
     */
    private $hasSubWallets;

    /**
     * @var string
     * @ORM\Column(name="local_balance", type="string", nullable=false)
     */
    private $localBalance;

    /**
     * @var Mutation[]|array
     * @ORM\OneToMany(targetEntity="IssetBV\PaymentBundle\Domain\Mutation", mappedBy="wallet")
     * @ORM\OrderBy(value={"dateTime" = "DESC"})
     */
    private $mutations;

    /**
     * @var string
     * @ORM\Column(name="reference", type="string", nullable=false)
     */
    private $reference;

    /**
     * This should be the primary key but we decided otherwise.
     *
     * @var string
     * @ORM\Column(name="remote_identifier", type="string", nullable=false, length=32, unique=true)
     */
    private $remoteIdentifier;

    /**
     * @var string|null
     * @ORM\Column(name="remote_status", type="string", nullable=true)
     */
    private $remoteStatus;

    /**
     * @var string
     * @ORM\Column(name="total_balance", type="string", nullable=false)
     */
    private $totalBalance;

    /**
     * @var TalosWalletType
     * @ORM\ManyToOne(targetEntity="IssetBV\TalosBundle\Entity\TalosWalletType")
     * @ORM\JoinColumn(name="wallet_type_id", referencedColumnName="id")
     */
    private $type;

    /**
     * @var TalosWalletOwnership[]|ArrayCollection
     * @ORM\OneToMany(targetEntity="IssetBV\TalosBundle\Entity\TalosWalletOwnership", mappedBy="wallet")
     */
    private $walletOwnerships;

    /**
     * TalosWallet constructor.
     *
     * @param TalosWalletType $walletType
     * @param string $remoteIdentifier
     * @param string $reference
     */
    public function __construct(TalosWalletType $walletType, string $remoteIdentifier, string $reference)
    {
        $this->initCrudFields();
        $this->type = $walletType;
        $this->remoteIdentifier = $remoteIdentifier;
        $this->reference = $reference;
        $this->totalBalance = '0.0000';
        $this->localBalance = '0.0000';
        $this->hasSubWallets = false;
        $this->walletOwnerships = new ArrayCollection();
    }

    /**
     * @return TalosAutoTopupProfile|null
     */
    public function getAutoTopupProfile()
    {
        return $this->autoTopupProfile;
    }

    /**
     * @param TalosAutoTopupProfile|null $autoTopupProfile
     */
    public function setAutoTopupProfile(TalosAutoTopupProfile $autoTopupProfile = null)
    {
        $this->autoTopupProfile = $autoTopupProfile;
    }

    /**
     * @return Money
     */
    public function getBalance(): Money
    {
        // @todo temporary logic for converting Talos' 4 decimal balance to Money
        return new Money(bcmul($this->totalBalance, '100', 0), new Currency('EUR'));
    }

    /**
     * @return string
     */
    public function getIdentifier(): string
    {
        return $this->remoteIdentifier;
    }

    /**
     * @return Mutation[]
     */
    public function getMutations()
    {
        return $this->mutations;
    }

    /**
     * Returns an empty optional if the wallet has no owner or multiple owners.
     *
     * @return Option<WalletOwner>
     */
    public function getOwner(): Option
    {
        // @todo refactor this to a more functional version
        if (1 === $this->walletOwnerships->count()) {
            return Option::ensure($this->walletOwnerships->map(
                function (TalosWalletOwnership $ownership) {
                    return $ownership->getOwner();
                }
            )->first());
        }

        return None::create();
    }

    /**
     * @return WalletOwner[]|Collection
     */
    public function getOwners()
    {
        return $this->walletOwnerships->map(
            function (TalosWalletOwnership $ownership) {
                return $ownership->getOwner();
            }
        );
    }

    /**
     * @return string
     */
    public function getReference(): string
    {
        return $this->reference;
    }

    /**
     * Returns the remote identifier of this object.
     *
     * @return Option
     */
    public function getRemoteIdentifier(): Option
    {
        return Option::ensure($this->remoteIdentifier);
    }

    /**
     * @param string $remoteIdentifier
     */
    public function setRemoteIdentifier(string $remoteIdentifier)
    {
        $this->remoteIdentifier = $remoteIdentifier;
    }

    /**
     * @return string|null
     */
    public function getRemoteStatus()
    {
        return $this->remoteStatus;
    }

    public function setRemoteStatus(string $remoteStatusCode)
    {
        $this->remoteStatus = $remoteStatusCode;
    }

    public function setAutoTopupPayment(Payment $autoTopupPayment = null)
    {
        $this->autoTopupPayment = $autoTopupPayment;
    }

    public function setBalance(string $balance)
    {
        $this->totalBalance = $balance;
    }

    /**
     * @param bool $hasSubWallets
     */
    public function setHasSubWallets(bool $hasSubWallets)
    {
        $this->hasSubWallets = $hasSubWallets;
    }

    /**
     * @param string $localBalance
     */
    public function setLocalBalance(string $localBalance)
    {
        $this->localBalance = $localBalance;
    }

    /**
     * @param string $reference
     */
    public function setReference($reference)
    {
        $this->reference = $reference;
    }

    /**
     * @param TalosWalletType $type
     */
    public function setType(TalosWalletType $type)
    {
        $this->type = $type;
    }

    /**
     * @return Option<AutoRenewConfiguration>
     */
    public function getAutoRenewConfiguration(): Option
    {
        $autoTopupPayment = Option::ensure($this->autoTopupPayment);
        $autoTopupProfile = Option::ensure($this->autoTopupProfile);

        return Option::ensure(
            $autoTopupPayment->isDefined() && $autoTopupProfile->isDefined()
                ? new AutoRenewConfigurationWrapper($autoTopupPayment->get(), $autoTopupProfile->get())
                : null
        );
    }

    public function getType(): WalletType
    {
        return $this->type;
    }
}
