<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Controller;

use IssetBV\Json\Exception\DecodeException;
use IssetBV\Json\JsonThing;
use IssetBV\Json\SimpleDecoder;
use IssetBV\TalosBundle\Gateway\Response\RequestObject;
use IssetBV\TalosBundle\Gateway\Response\Response as ApiResponse;
use IssetBV\TalosBundle\Gateway\Response\Response as GatewayResponse;
use IssetBV\TalosBundle\ResponseHandler\Factory\PaymentResponseHandlerFactory;
use IssetBV\TalosBundle\ResponseHandler\PaymentResponseHandler;
use IssetBV\TalosBundle\ResponseHandler\WalletResponseHandler;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use function Functional\each;

/**
 * Class NotificationController.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class NotificationController extends Controller
{
    /**
     * @Route(path="talos/notification")
     *
     * @param Request $request
     *
     * @return Response
     */
    public function receiveAction(Request $request)
    {
        $this->logRequest($request);
        $objectManager = $this->get('doctrine.orm.entity_manager');

        try {
            $response = ApiResponse::fromJsonThing($this->decodeResponse($request));

            $this->processResponse($response);

            $objectManager->flush();
        } catch (DecodeException $e) {
            throw new BadRequestHttpException('Invalid JSON: ' . $e->getMessage());
        }

        return new Response('');
    }

    /**
     * @param Request $request
     */
    protected function logRequest(Request $request)
    {
        $log = 'Method: ' . $request->getMethod() . PHP_EOL;
        $log .= 'Endpoint: ' . $request->getUri() . PHP_EOL;

        $log .= PHP_EOL . 'Headers: ' . PHP_EOL . json_encode($request->headers->all(), JSON_PRETTY_PRINT) . PHP_EOL;
        $log .= PHP_EOL . 'Data: ' . PHP_EOL . json_encode($request->request->all(), JSON_PRETTY_PRINT) . PHP_EOL;

        $this->get('monolog.logger.payment')->error($log);
    }

    /**
     * @param $response
     */
    protected function processResponse(GatewayResponse $response)
    {
        each($response->getRequestObjects(),
            function (RequestObject $requestObject) use ($response) {
                if ('Payment' === $requestObject->getType()) {
                    /** @var PaymentResponseHandler $paymentResponseHandler */
                    $paymentResponseHandler = $this->get(PaymentResponseHandlerFactory::class)->createResponseHandler();

                    $paymentResponseHandler->handleResponse($response);
                } elseif ('Wallet' === $requestObject->getType()) {
                    $walletResponseHandler = $this->get(WalletResponseHandler::class);
                    $walletResponseHandler->handleResponse($response);
                }
            }
        );
    }

    private function decodeResponse(Request $request): JsonThing
    {
        $data = (new SimpleDecoder())->decode((string) $request->getContent());

        if ($data instanceof JsonThing) {
            return $data;
        }

        throw new \RuntimeException('Cannot handle response because it\'s not an object');
    }
}
