<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\ExecutePayment;

use Exception;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Payment\PaymentType;
use IssetBV\PaymentBundle\Domain\PaymentMethod;
use IssetBV\TalosBundle\Gateway\Exception\ResponseException;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
final class CannotExecutePaymentException extends Exception
{
    public static function becauseNoIssuerWasSet($paymentIdentifier = 'unknown'): self
    {
        return new self("Unable to create service request for ideal payment (id: $paymentIdentifier) because no payment issuer (bank) was set in the payment");
    }

    public static function becauseTheReturnUrlWasNotSet($paymentIdentifier = 'unknown'): self
    {
        return new self("Unable to execute payment (id: $paymentIdentifier) because the return URL was not set in the payment");
    }

    public static function becauseThePaymentDoesNotExist(LocalPaymentIdentifier $paymentId): self
    {
        return new self('Unable to execute payment because no payment with id ' . $paymentId->asString() . ' could be found');
    }

    public static function becauseThePaymentTypeIsNotSupported(PaymentType $type): self
    {
        return new self('Unable to execute payment because the payment type \'' . (string) $type . '\' is not supported');
    }

    public static function becauseNoOriginalPaymentIsSet(): self
    {
        return new self('Unable to execute payment because no original payment is set or no remote identifier exists for original payment');
    }

    public static function becauseOfAResponseException(ResponseException $e)
    {

        return new self('Unable to execute payment because the response received from the gateway while executing the payment was invalid', 0, $e);
    }

    public static function becauseThePaymentMethodIsNotActive(PaymentMethod $paymentMethod)
    {
        return new self(sprintf('The payment method %s is not active until %s', $paymentMethod->getDisplayName(), $paymentMethod->getDateActive()->format('c')));
    }
}
