<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use DateTime;
use IssetBV\PaymentBundle\Cli\Renderer\SubscriptionRenderer;
use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\NullOutput;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\OutputStyle;
use Symfony\Component\Console\Style\SymfonyStyle;
use function Functional\map;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class ShowAllSubscriptionsCommand extends Command
{
    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;
    /**
     * @var SubscriptionRenderer
     */
    private $subscriptionRenderer;
    /**
     * @var CommandBus
     */
    private $commandBus;

    public function __construct(SubscriptionRepository $subscriptionRepository, SubscriptionRenderer $subscriptionRenderer, CommandBus $commandBus)
    {
        parent::__construct('subscription:show');
        $this->subscriptionRepository = $subscriptionRepository;
        $this->subscriptionRenderer = $subscriptionRenderer;
        $this->commandBus = $commandBus;
    }

    public function configure()
    {
        $this->addArgument('subscription', InputArgument::REQUIRED, 'The id of the subscription you want to show');
        $this->addOption('denormalize', 'd', InputOption::VALUE_NONE, 'Pass this option if you also want to denormalize the subscription first');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $style = new SymfonyStyle($input, $output);
        $identifier = new SubscriptionIdentifier($input->getArgument('subscription'));

        if ($input->getOption('denormalize')) {
            $style->note('Denormalizing data');
            $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($identifier));
        }

        $subscription = $this->subscriptionRepository
            ->optionallyFind($identifier);

        if ($subscription->isEmpty()) {
            $style->error('Subscription ' . $identifier . ' could not be found');
        } else {
            $this->subscriptionRenderer->render($subscription->get(), $output);
        }

    }

    public function formatDateTime(DateTime $dateTime)
    {
        return $dateTime->format(DateTime::W3C);
    }
}
