<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\RenewSubscription;

use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Identifier\UuidPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Invoice\Invoice;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\RenewAmountProvider;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use League\Tactician\CommandBus;
use Money\Currency;
use Money\Money;
use PhpOption\None;
use PhpOption\Option;
use PHPUnit\Framework\TestCase;
use Prophecy\Argument;
use Ramsey\Uuid\Uuid;

class RenewSubscriptionHandlerTest extends TestCase
{

    public function testThatNonExpiredSubscriptionShouldNotRenew()
    {
        $commandBus = $this->prophesize(CommandBus::class);
        $invoiceFactory = $this->prophesize(InvoiceFactory::class);
        $paymentFactory = $this->prophesize(PaymentFactory::class);

        $subscriptionStatus = $this->prophesize(SubscriptionStatus::class);
        $subscriptionStatus->isExpired()->shouldBeCalled()->willReturn(false);

        $subscriptionIdentifier = $this->prophesize(SubscriptionIdentifier::class);
        $subscriptionIdentifier->asString()->willReturn('xy32');

        $subscription = $this->prophesize(Subscription::class);
        $subscription->getStatus()->willReturn($subscriptionStatus);
        $subscription->isCanceled()->willReturn(false);
        $subscription->getId()->willReturn($subscriptionIdentifier);

        $subscriptionRepository = $this->prophesize(SubscriptionRepository::class);
        $subscriptionRepository->optionallyFind(Argument::exact($subscriptionIdentifier))->willReturn(Option::fromValue($subscription->reveal()));

        $renewAmountProvider = $this->prophesize(RenewAmountProvider::class);

        $renewSubscriptionHandler = new RenewSubscriptionHandler(
            $commandBus->reveal(),
            $invoiceFactory->reveal(),
            $paymentFactory->reveal(),
            $subscriptionRepository->reveal(),
            $renewAmountProvider->reveal()
        );

        $this->expectException(CannotRenewSubscriptionException::class);
        $this->expectExceptionMessage('Cannot renew subscription xy32 because it is not expired');

        $renewSubscriptionHandler->handle(new RenewSubscriptionCommand($subscriptionIdentifier->reveal()));
    }

    public function testThatItWontRenewCanceledACanceledSubscription()
    {
        $commandBus = $this->prophesize(CommandBus::class);
        $invoiceFactory = $this->prophesize(InvoiceFactory::class);
        $paymentFactory = $this->prophesize(PaymentFactory::class);

        $subscriptionStatus = $this->prophesize(SubscriptionStatus::class);
        $subscriptionStatus->isExpired()->shouldBeCalled()->willReturn(true);

        $subscriptionIdentifier = $this->prophesize(SubscriptionIdentifier::class);
        $subscriptionIdentifier->asString()->willReturn('xy32');

        $subscription = $this->prophesize(Subscription::class);
        $subscription->getStatus()->willReturn($subscriptionStatus);
        $subscription->isCanceled()->shouldBeCalled()->willReturn(true);
        $subscription->getId()->willReturn($subscriptionIdentifier);

        $subscriptionRepository = $this->prophesize(SubscriptionRepository::class);
        $subscriptionRepository->optionallyFind(Argument::exact($subscriptionIdentifier))->willReturn(Option::fromValue($subscription->reveal()));

        $renewAmountProvider = $this->prophesize(RenewAmountProvider::class);

        $renewSubscriptionHandler = new RenewSubscriptionHandler(
            $commandBus->reveal(),
            $invoiceFactory->reveal(),
            $paymentFactory->reveal(),
            $subscriptionRepository->reveal(),
            $renewAmountProvider->reveal()
        );

        $this->expectException(CannotRenewSubscriptionException::class);
        $this->expectExceptionMessage("Cannot renew subscription xy32 because it has been canceled");

        $renewSubscriptionHandler->handle(new RenewSubscriptionCommand($subscriptionIdentifier->reveal()));
    }


    public function testThatItThrowsAnExceptionIfNoRenewablePaymentIsFound()
    {
        $commandBus = $this->prophesize(CommandBus::class);
        $invoiceFactory = $this->prophesize(InvoiceFactory::class);
        $paymentFactory = $this->prophesize(PaymentFactory::class);

        $subscriptionStatus = $this->prophesize(SubscriptionStatus::class);
        $subscriptionStatus->isExpired()->shouldBeCalled()->willReturn(true);

        $subscriptionIdentifier = $this->prophesize(SubscriptionIdentifier::class);
        $subscriptionIdentifier->asString()->willReturn('xy32');

        $subscription = $this->prophesize(Subscription::class);
        $subscription->getStatus()->willReturn($subscriptionStatus);
        $subscription->isCanceled()->shouldBeCalled()->willReturn(false);
        $subscription->getId()->willReturn($subscriptionIdentifier);
        $subscription->findRenewablePayment()->willReturn(None::create());

        $subscriptionRepository = $this->prophesize(SubscriptionRepository::class);
        $subscriptionRepository->optionallyFind(Argument::exact($subscriptionIdentifier))->willReturn(Option::fromValue($subscription->reveal()));

        $renewAmountProvider = $this->prophesize(RenewAmountProvider::class);

        $renewSubscriptionHandler = new RenewSubscriptionHandler(
            $commandBus->reveal(),
            $invoiceFactory->reveal(),
            $paymentFactory->reveal(),
            $subscriptionRepository->reveal(),
            $renewAmountProvider->reveal()
        );

        $this->expectException(CannotRenewSubscriptionException::class);
        $this->expectExceptionMessage('Cannot renew subscription because the subscription with id xy32 does not have a valid (repeatable) payment');

        $renewSubscriptionHandler->handle(new RenewSubscriptionCommand($subscriptionIdentifier->reveal()));
    }

    public function testThatItRenewsTheSubscription()
    {
        $commandBus = $this->prophesize(CommandBus::class);

        $invoice = $this->prophesize(Invoice::class);

        $invoiceFactory = $this->prophesize(InvoiceFactory::class);
        $invoiceFactory->createInvoice()->shouldBeCalled()->willReturn($invoice);

        $executablePayment = $this->prophesize(ExecutablePayment::class);
        $executablePayment->getId()->willReturn(new UuidPaymentIdentifier(Uuid::uuid4()));

        $paymentFactory = $this->prophesize(PaymentFactory::class);
        $paymentFactory->createRecurringPayment(Argument::any(), Argument::any(), Argument::any())->shouldBeCalled()->willReturn($executablePayment);

        $subscriptionStatus = $this->prophesize(SubscriptionStatus::class);
        $subscriptionStatus->isExpired()->shouldBeCalled()->willReturn(true);

        $subscriptionIdentifier = $this->prophesize(SubscriptionIdentifier::class);
        $subscriptionIdentifier->asString()->willReturn('xy32');

        $renewablePayment = $this->prophesize(ExecutablePayment::class);
        $renewablePayment->getId()->willReturn(new UuidPaymentIdentifier(Uuid::uuid4()));

        $subscription = $this->prophesize(Subscription::class);
        $subscription->getStatus()->willReturn($subscriptionStatus);
        $subscription->isCanceled()->shouldBeCalled()->willReturn(false);
        $subscription->getId()->willReturn($subscriptionIdentifier);
        $subscription->findRenewablePayment()->willReturn(Option::fromValue($renewablePayment->reveal()));
        $subscription->createNewTerm(Argument::exact($executablePayment))->shouldBeCalled();

        $subscriptionRepository = $this->prophesize(SubscriptionRepository::class);
        $subscriptionRepository->optionallyFind(Argument::exact($subscriptionIdentifier))->willReturn(Option::fromValue($subscription->reveal()));

        $renewAmountProvider = $this->prophesize(RenewAmountProvider::class);
        $renewAmountProvider->findRenewAmount(Argument::exact($subscription))->shouldBeCalled()->willReturn(new Money(100, new Currency('EUR')));

        $renewSubscriptionHandler = new RenewSubscriptionHandler(
            $commandBus->reveal(),
            $invoiceFactory->reveal(),
            $paymentFactory->reveal(),
            $subscriptionRepository->reveal(),
            $renewAmountProvider->reveal()
        );

        $renewSubscriptionHandler->handle(new RenewSubscriptionCommand($subscriptionIdentifier->reveal()));
    }
}
