<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\ResponseHandler;

use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Invoice\DefaultInvoiceNumber;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Payment\Event\PaymentStatusChangedEvent;
use IssetBV\PaymentBundle\Domain\RemoteObject;
use IssetBV\PaymentBundle\Domain\Repository\PaymentIssuerRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentMethodRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\Gateway\Response\RequestObject;
use IssetBV\TalosBundle\Gateway\Response\Response;
use IssetBV\TalosBundle\Gateway\Response\Status;
use IssetBV\TalosBundle\Gateway\Shared\GroupField;
use IssetBV\TalosBundle\Mapper\Exception\MapperException;
use IssetBV\TalosBundle\Storage\EntityStore;
use Money\MoneyParser;
use PhpOption\None;
use PhpOption\Option;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class PaymentResponseHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class PaymentResponseHandler implements ResponseHandler
{
    /**
     * @var EntityStore
     */
    private $entityStore;

    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var MoneyParser
     */
    private $moneyParser;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var PaymentIssuerRepository
     */
    private $paymentIssuerRepository;

    /**
     * @var PaymentMethodRepository
     */
    private $paymentMethodRepository;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * PaymentResponseHandler constructor.
     *
     * @param EntityStore $entityStore
     * @param InvoiceFactory $invoiceFactory
     * @param MoneyParser $moneyParser
     * @param PaymentFactory $paymentFactory
     * @param PaymentIssuerRepository $paymentIssuerRepository
     * @param PaymentMethodRepository $paymentMethodRepository
     * @param PaymentRepository $paymentRepository
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        EntityStore $entityStore,
        InvoiceFactory $invoiceFactory,
        MoneyParser $moneyParser,
        PaymentFactory $paymentFactory,
        PaymentIssuerRepository $paymentIssuerRepository,
        PaymentMethodRepository $paymentMethodRepository,
        PaymentRepository $paymentRepository,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->entityStore = $entityStore;
        $this->invoiceFactory = $invoiceFactory;
        $this->moneyParser = $moneyParser;
        $this->paymentFactory = $paymentFactory;
        $this->paymentIssuerRepository = $paymentIssuerRepository;
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->paymentRepository = $paymentRepository;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param Response $response
     *
     * @throws MapperException
     */
    public function handleResponse(Response $response)
    {
        $responseStatus = $response->findStatus()->getOrElse(null);

        if ($responseStatus instanceof Status) {
            $code = $responseStatus->getCode();

            if ('ValidationError' === $code || 'Failed' === $code) {
                throw MapperException::requestError($code, $responseStatus->getValue());
            }

            if ('UserRecoverableValidationError' === $code) {
                throw MapperException::requestError($code, $responseStatus->getValue());
            }
        }

        /** @var RequestObject $requestObject */
        $requestObject = $response->getOneRequestObjectWithType('Payment')->getOrThrow(MapperException::propertyNotFound('Payment'));

        $payment = $this->ensurePayment($requestObject);

        $oldStatus = null;
        $newStatus = null;

        if ($payment instanceof RemoteObject) {
            $oldStatus = $payment->getRemoteStatus();

            $this->processRemoteObject($requestObject, $payment);

            $newStatus = $payment->getRemoteStatus();
        }

        if ($payment instanceof ExecutablePayment) {
            if (null !== $oldStatus && null !== $newStatus && $oldStatus !== $newStatus) {
                $this->eventDispatcher->dispatch(PaymentStatusChangedEvent::name(), new PaymentStatusChangedEvent($payment->getId(), $oldStatus, $newStatus));
            }

            $this->processExecutablePayment($response, $payment);
        }
    }

    /**
     * @param RequestObject $requestObject
     *
     * @return Payment
     */
    private function ensurePayment(RequestObject $requestObject): Payment
    {
        return $this
            ->findPayment($requestObject)
            ->getOrCall(function () use ($requestObject) {
                return $this->createPayment($requestObject);
            });
    }

    /**
     * @param RequestObject $requestObject
     *
     * @return Option
     */
    private function findPayment(RequestObject $requestObject): Option
    {
        return $this->paymentRepository
            ->findOneByRemoteIdentifier($requestObject->getIdentifier())
            ->orElse($this->paymentRepository->findOneByReference($requestObject->getReference()->getOrElse('')))
            ->orElse(None::create());
    }

    /**
     * @param RequestObject $requestObject
     *
     * @return Payment
     */
    private function createPayment(RequestObject $requestObject): Payment
    {
        $invoice = $requestObject->getSingleFieldValue('InvoiceNumber')
            ->map(function (string $invoiceNumber) {
                return $this->invoiceFactory->createInvoice(new DefaultInvoiceNumber($invoiceNumber));
            })
            ->getOrCall(function () {
                return $this->invoiceFactory->createInvoice();
            });

        $paymentIssuer = $requestObject->getSingleFieldValue('CustomerAccountIssuer')
            ->flatMap(function (string $customerAccountIssuer) {
                return $this->paymentIssuerRepository->findOneByCode($customerAccountIssuer);
            })
            ->getOrElse(null);

        $paymentMethod = $requestObject->getSingleFieldValue('PaymentMethod')
            ->flatMap(function (string $paymentMethod) {
                return $this->paymentMethodRepository->findByServiceName($paymentMethod);
            })
            ->getOrThrow(MapperException::propertyNotFound('PaymentMethod'));

        $payment = $this->paymentFactory->createPayment(
            $invoice,
            $this->moneyParser->parse($requestObject->getSingleFieldValue('AmountDebit')->getOrThrow(MapperException::propertyNotFound('AmountDebit'))),
            $paymentMethod,
            $paymentIssuer,
            $requestObject->getSingleFieldValue('RecurrentType')->getOrThrow(MapperException::propertyNotFound('RecurrentType'))
        );

        $this->entityStore->persist($invoice);
        $this->entityStore->persist($payment);

        return $payment;
    }

    /**
     * @param Response $response
     * @param ExecutablePayment $payment
     */
    private function processExecutablePayment(Response $response, ExecutablePayment $payment)
    {
        foreach ($response->getServices() as $service) {
            $service->getField('RequiredAction')
                ->flatMap(function (GroupField $requiredAction) {
                    return $requiredAction->getSingleFieldValue('RedirectURL');
                })
                ->forAll(function (string $redirectUrl) use ($payment) {
                    $payment->setPaymentUrl($redirectUrl);
                });
        }

        $requestObject = $response->getOneRequestObjectWithType('Payment')->getOrThrow(MapperException::propertyNotFound('Payment'));

        $requestObject->getSingleFieldValue('CustomerAccountName')
            ->forAll(function (string $value) use ($payment) {
                $payment->setCustomerAccountName($value);
            });

        $requestObject->getSingleFieldValue('CustomerAccountIdentifier')
            ->forAll(function (string $value) use ($payment) {
                $payment->setCustomerAccountIdentifier($value);
            });
    }

    /**
     * @param RequestObject $requestObject
     * @param RemoteObject $remoteObject
     */
    private function processRemoteObject(RequestObject $requestObject, RemoteObject $remoteObject)
    {
        $remoteObject->setRemoteIdentifier($requestObject->getIdentifier());

        $requestObject->getField('Status')
            ->flatMap(function (GroupField $groupField) {
                return $groupField->getSingleFieldValue('StatusCode');
            })
            ->forAll(function (string $statusCode) use ($remoteObject) {
                $remoteObject->setRemoteStatus($statusCode);
            });
    }
}
