<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\ResponseHandler\Factory;

use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentIssuerRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentMethodRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use IssetBV\TalosBundle\Entity\TalosPaymentMethod;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\ResponseHandler\CreditCardPaymentResponseHandler;
use IssetBV\TalosBundle\ResponseHandler\PaymentResponseHandler;
use IssetBV\TalosBundle\Storage\EntityStore;
use Money\MoneyParser;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\PropertyAccess\PropertyAccess;

class PaymentResponseHandlerFactory
{
    /**
     * @var EntityStore
     */
    private $entityStore;

    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var MoneyParser
     */
    private $moneyParser;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var PaymentIssuerRepository
     */
    private $paymentIssuerRepository;

    /**
     * @var PaymentMethodRepository
     */
    private $paymentMethodRepository;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * PaymentResponseHandler constructor.
     *
     * @param EntityStore $entityStore
     * @param InvoiceFactory $invoiceFactory
     * @param MoneyParser $moneyParser
     * @param PaymentFactory $paymentFactory
     * @param PaymentIssuerRepository $paymentIssuerRepository
     * @param PaymentMethodRepository $paymentMethodRepository
     * @param PaymentRepository $paymentRepository
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        EntityStore $entityStore,
        InvoiceFactory $invoiceFactory,
        MoneyParser $moneyParser,
        PaymentFactory $paymentFactory,
        PaymentIssuerRepository $paymentIssuerRepository,
        PaymentMethodRepository $paymentMethodRepository,
        PaymentRepository $paymentRepository,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->entityStore = $entityStore;
        $this->invoiceFactory = $invoiceFactory;
        $this->moneyParser = $moneyParser;
        $this->paymentFactory = $paymentFactory;
        $this->paymentIssuerRepository = $paymentIssuerRepository;
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->paymentRepository = $paymentRepository;
        $this->eventDispatcher = $eventDispatcher;
    }

    public function createResponseHandler()
    {
        return new PaymentResponseHandler(
            $this->entityStore,
            $this->invoiceFactory,
            $this->moneyParser,
            $this->paymentFactory,
            $this->paymentIssuerRepository,
            $this->paymentMethodRepository,
            $this->paymentRepository,
            $this->eventDispatcher
        );
    }

    /**
     * @param Payment $payment
     *
     * @return ResponseHandler
     */
    public function createResponseHandlerPayment(Payment $payment)
    {
        //@todo I'm sure we can come up with a better way to implement this.

        $serviceName = PropertyAccess::createPropertyAccessor()->getValue($payment->getPaymentMethod(), 'serviceName');

        if (in_array($serviceName, [TalosPaymentMethod::SERVICE_NAME_MASTER_CARD, TalosPaymentMethod::SERVICE_NAME_AMEX, TalosPaymentMethod::SERVICE_NAME_VISA, TalosPaymentMethod::SERVICE_NAME_BANCONTACT], true)) {
            return new CreditCardPaymentResponseHandler($this->asExecutablePayment($payment));
        }

        return $this->createResponseHandler();
    }

    /**
     * @param Payment $payment
     *
     * @return ExecutablePayment
     */
    private function asExecutablePayment(Payment $payment): ExecutablePayment
    {
        if ($payment instanceof ExecutablePayment) {
            return $payment;
        }

        throw new \RuntimeException('Cannot use Payment as ExecutablePayment');
    }
}
