<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Entity;

use DateInterval;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\Exception\InvalidPaymentIntervalException;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription as SubscriptionInterface;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\TalosBundle\Storage\CreatedUpdatedFields;
use PhpOption\Option;
use function Functional\filter;
use function Functional\first;

/**
 * @author Tim Fennis <tim@isset.nl>
 *
 * @ORM\Entity(repositoryClass="IssetBV\PaymentBundle\Repository\DoctrineSubscriptionRepository")
 * @ORM\Table(name="subscriptions")
 * @ORM\HasLifecycleCallbacks()
 */
class Subscription implements SubscriptionInterface
{
    use CreatedUpdatedFields;

    /**
     * @var int
     * @ORM\Id()
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var int
     * @ORM\Column(name="denormalized_status", type="integer", nullable=false)
     */
    private $denormalizedStatus;

    /**
     * @var DateTime
     * @ORM\Column(name="start_date", type="datetime", nullable=false)
     */
    private $startDate;

    /**
     * @var DateTime
     * @ORM\Column(name="date_canceled", type="datetime", nullable=true)
     */
    private $dateCanceled;

    /**
     * @var Interval
     * @ORM\Embedded(class="Interval")
     */
    private $paymentInterval;

    /**
     * @var ArrayCollection
     * @ORM\OneToMany(targetEntity="IssetBV\PaymentBundle\Entity\SubscriptionTerm", mappedBy="subscription", cascade={"all"})
     */
    private $subscriptionTerms;

    /**
     * @var Payment|null
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment")
     * @ORM\JoinColumn(name="original_payment_id")
     */
    private $originalPayment;

    /**
     * @var DateTime|null
     * @ORM\Column(name="d_expiration_date", type="datetime", nullable=true)
     */
    private $denormalizedExpirationDate;

    private function __construct(Interval $paymentInterval, DateTime $startDate)
    {
        $this->initCrudFields();
        $this->paymentInterval = $paymentInterval;
        $this->startDate = $startDate;
        $this->subscriptionTerms = new ArrayCollection();

        // Set the initial status to pending
        $this->updateDenormalizedStatus(SubscriptionStatus::pending());
    }

    public static function createSubscription(Interval $paymentInterval, DateTime $startDate, Payment $initialPayment = null, Interval $initialInterval = null): self
    {
        $initialInterval = Option::fromValue($initialInterval)->getOrElse($paymentInterval);

        $subscription = new self($paymentInterval, $startDate);
        $subscription->createFirstTerm($initialPayment, $initialInterval);

        return $subscription;
    }

    public function createNewTerm(Payment $newPayment)
    {
        $this->subscriptionTerms->add(new SubscriptionTerm(
            $this,
            $this->getDateValidUntil()->getOrElse($this->getStartDate()),
            $newPayment,
            $this->paymentInterval,
            false
        ));

        $this->updateDenormalizedExpirationDate();
    }

    public function createFreeTerm(DateInterval $dateInterval)
    {
        $this->subscriptionTerms->add(new SubscriptionTerm(
            $this,
            $this->getDateValidUntil()->getOrElse($this->getStartDate()),
            null,
            Interval::createFromDateInterval($dateInterval),
            true
        ));

        $this->updateDenormalizedExpirationDate();
    }

    /**
     * If you set the payment to NULL the first term will have activityOverride = true.
     *
     * @param Payment|null $payment The repeatable payment for the first term
     * @param Interval $interval The interval of the first term
     */
    public function createFirstTerm(Payment $payment = null, Interval $interval)
    {
        $this->originalPayment = $payment;
        $this->subscriptionTerms->add(new SubscriptionTerm(
            $this,
            $this->getStartDate(),
            $this->originalPayment,
            $interval,
            $payment ? false : true
        ));

        $this->updateDenormalizedExpirationDate();
    }

    public function getOriginalPayment(): Option
    {
        return Option::fromValue($this->originalPayment);
    }

    public function isCanceled(): bool
    {
        return null !== $this->dateCanceled;
    }

    /**
     * Returns a collection compatible with `Collection` and `Selectable`.
     *
     * @return Collection|Payment[]
     */
    public function getPayments()
    {
        return $this->getSubscriptionTerms()
            ->filter(function (SubscriptionTerm $subscriptionTerm) {
                return $subscriptionTerm->getPayment()->isDefined();
            })
            ->map(function (SubscriptionTerm $subscriptionTerm) {
                return $subscriptionTerm->getPayment()->get();
            });
    }

    /**
     * @return SubscriptionTerm[]|ArrayCollection
     */
    public function getSubscriptionTerms()
    {
        return $this->subscriptionTerms;
    }

    public function getId(): SubscriptionIdentifier
    {
        return new SubscriptionIdentifier($this->id);
    }

    public function getStartDate(): DateTime
    {
        return $this->startDate;
    }

    /**
     * @throws InvalidPaymentIntervalException if the database has somehow gotten corrupted
     *
     * @return DateInterval
     */
    public function getPaymentInterval(): DateInterval
    {
        return $this->paymentInterval->getDateInterval()
            ->getOrThrow(new InvalidPaymentIntervalException());
    }

    public function getDateValidUntil(): Option
    {
        $newestTermDate = null;
        $newestTerm = null;

        foreach ($this->getSubscriptionTerms() as $subscriptionTerm) {
            if (null === $newestTermDate) {
                $newestTermDate = $subscriptionTerm->getDateTime();
                $newestTerm = $subscriptionTerm;
            } elseif (null !== $newestTerm && $subscriptionTerm->getDateTime()->getTimestamp() > $newestTerm->getDateTime()->getTimestamp()) {
                $newestTermDate = $subscriptionTerm->getDateTime();
                $newestTerm = $subscriptionTerm;
            }
        }

        return Option::fromValue(null !== $newestTerm ? $newestTerm->getEndDate() : null);
    }

    public function cancel(): bool
    {
        return null === $this->dateCanceled
            ? (bool) ($this->dateCanceled = new DateTime())
            : false;
    }

    public function updateDenormalizedStatus(SubscriptionStatus $subscriptionStatus)
    {
        $this->denormalizedStatus = $subscriptionStatus->getCode();
    }

    public function updateDenormalizedExpirationDate()
    {
        $this->denormalizedExpirationDate = $this->getDateValidUntil()->getOrElse(null);
    }

    public function getStatus(): SubscriptionStatus
    {
        return SubscriptionStatus::safelyFromInteger($this->denormalizedStatus);
    }

    public function getDateCanceled(): Option
    {
        return Option::ensure($this->dateCanceled);
    }

    /**
     * @return Option<ExecutablePayment> an Option containing an ExecutablePayment
     */
    public function findRenewablePayment(): Option
    {
        $executablePayments = filter($this->getPayments(), function ($payment) {
            return $payment instanceof ExecutablePayment;
        });

        $paymentsWithFirstType = filter($executablePayments, function (ExecutablePayment $payment) {
            return $payment->getType()->isFirst();
        });

        return Option::fromValue(first($paymentsWithFirstType));
    }
}
