<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use Doctrine\Common\Collections\Criteria;
use Exception;
use IssetBV\PaymentBundle\Cli\Renderer\SubscriptionRenderer;
use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\CannotRenewSubscriptionException;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\RenewSubscriptionCommand;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use function Functional\each;
use function Functional\filter;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionsCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SymfonyStyle
     */
    private $console;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    /**
     * @var SubscriptionRenderer
     */
    private $subscriptionRenderer;

    public function __construct(
        CommandBus $commandBus,
        SubscriptionRepository $subscriptionRepository,
        SubscriptionRenderer $subscriptionRenderer
    ) {
        parent::__construct('subscription:renew');
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
        $this->subscriptionRenderer = $subscriptionRenderer;
    }

    public function configure()
    {
        // do nothing for now
        $this->addOption('force', 'f', InputOption::VALUE_NONE, 'Unless you specify this option no payments will be executed');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->console = new SymfonyStyle($input, $output);
        $this->console->title('Renewing subscriptions');
        $this->renewSubscriptions(false === $input->getOption('force'));
    }

    private function renewSubscriptions(bool $mock = true)
    {
        $this->console->text('Searching for subscriptions that are both expired and not canceled');

        $subscriptions = $this->findSubscriptionsToUpdate();

        $this->console->text('Updating the status of the expired subscriptions');

        each($subscriptions, function (Subscription $subscription) {
            $this->commandBus->handle(DenormalizeSubscriptionStatusCommand::fromSubscription($subscription));
        });

        $this->console->text('Rerunning search for subscriptions that are both expired and not canceled');

        $subscriptions = $this->findSubscriptionsToUpdate();

        $this->console->text(sprintf('Found %d subscriptions', count($subscriptions)));

        try {
            if (false === $mock) {
                $this->console->progressStart(count($subscriptions));

                each($subscriptions, function (Subscription $subscription) {
                    try {
                        $this->console->text('Renewing subscription ' . $subscription->getId());
                        $this->commandBus->handle(new RenewSubscriptionCommand($subscription->getId()));
                    } catch (CannotRenewSubscriptionException $e) {
                        $this->console->warning('CannotRenewSubscriptionException while renewing ' . $subscription->getId()->asString() . ' with message: ' . $e->getMessage());
                    }

                    $this->console->progressAdvance(1);
                });

                $this->console->progressFinish();
                $this->console->success('Subscriptions updated');
            } else {
                if (count($subscriptions) > 0) {
                    $this->console->section('Would renew the following subscriptions');
                }

                each($subscriptions, function (Subscription $subscription) {
                    $this->subscriptionRenderer->renderWithStyle($subscription, $this->console);
                });
            }
        } catch (Exception $e) {
            $this->console->progressFinish();
            $this->console->caution('Exception: ' . $e->getMessage());

            throw $e;
        }
    }

    /**
     * @return Subscription[]
     */
    private function findSubscriptionsToUpdate()
    {
        $criteria = Criteria::create();
        $criteria->andWhere(Criteria::expr()->eq('denormalizedStatus', SubscriptionStatus::expired()->getCode()));
        $criteria->andWhere(Criteria::expr()->eq('dateCanceled', null));

        $subscriptions = $this->subscriptionRepository->matching($criteria);

        return filter($subscriptions, function (Subscription $subscription) {
            return $subscription->getOriginalPayment()->isDefined() && $subscription->findRenewablePayment()->isDefined();
        });
    }
}
