<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus;

use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Event\SubscriptionStatusChangedEvent;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Entity\SubscriptionTerm;
use IssetBV\TalosBundle\Domain\TalosStatus;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use function Functional\filter;
use function Functional\flat_map;
use function Functional\some;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class DenormalizeSubscriptionStatusHandler
{
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(SubscriptionRepository $subscriptionRepository, EventDispatcher $eventDispatcher)
    {
        $this->subscriptionRepository = $subscriptionRepository;
        $this->eventDispatcher = $eventDispatcher;
    }

    public function handle(DenormalizeSubscriptionStatusCommand $command)
    {
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());
        $originalStatus = $subscription->getStatus();
        $subscription->updateDenormalizedStatus($this->determineStatus($subscription));
        $newStatus = $subscription->getStatus();

        // delta
        if (false === $originalStatus->equals($newStatus)) {
            $this->eventDispatcher->dispatch(
                SubscriptionStatusChangedEvent::name(),
                new SubscriptionStatusChangedEvent($subscription->getId(), $originalStatus, $newStatus)
            );
        }
    }

    protected function determineStatus(Subscription $subscription): SubscriptionStatus
    {
        $links = $subscription->getSubscriptionTerms();

        if ($subscription->getOriginalPayment()->isDefined()) {
            /** @var Payment $originalPayment */
            $originalPayment = $subscription->getOriginalPayment()->get();
            if ($originalPayment->getRemoteStatus() !== TalosStatus::SUCCESS) {
                return SubscriptionStatus::invalidated();
            }
        }

        $payments = flat_map($links, function (SubscriptionTerm $subscriptionTerm) {
            return $subscriptionTerm->getPayment()->getOrElse([]);
        });

        $someAreCanceled = some($payments, function (Payment $payment) {
            return TalosStatus::CANCELED_BY_USER === $payment->getRemoteStatus() ||
                TalosStatus::CANCELED_BY_BACK_OFFICE === $payment->getRemoteStatus() ||
                TalosStatus::EXPIRED === $payment->getRemoteStatus();
        });

        if ($someAreCanceled) {
            return SubscriptionStatus::invalidated();
        }

        $activeLinks = filter($links, function (SubscriptionTerm $subscriptionTerm) {
            return $subscriptionTerm->isActive();
        });

        $validLinks = filter($activeLinks, function (SubscriptionTerm $subscriptionTerm) {
            return $subscriptionTerm->isValid();
        });

        if (0 === count($links)) {
            return SubscriptionStatus::pending();
        }

        if (0 === count($activeLinks)) {
            return SubscriptionStatus::expired();
        }

        if (0 === count($validLinks)) {
            return SubscriptionStatus::pending();
        }

        return SubscriptionStatus::valid();
    }

    private function findSubscription($identifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($identifier)
            ->getOrThrow(CannotDenormalizeSubscriptionStatusException::becauseTheSubscriptionCouldNotBeFound($identifier));
    }
}
