<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway\Request;

use IssetBV\Json\JsonThing;
use IssetBV\TalosBundle\Gateway\Shared\Field;
use IssetBV\TalosBundle\Gateway\Shared\FieldContainer;
use JsonSerializable;

/**
 * Class ServiceBase.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class Service implements JsonSerializable
{
    use FieldContainer;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $action;

    /**
     * @var string
     */
    private $version;

    /**
     * ServiceBase constructor.
     *
     * @param string $name
     * @param string $action
     * @param array $fields
     */
    public function __construct(string $name, string $action, array $fields = [])
    {
        $this->name = $name;
        $this->action = $action;
        $this->version = '1';
        $this->setFields($fields);
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getAction(): string
    {
        return $this->action;
    }

    /**
     * @param JsonThing $jsonThing
     *
     * @return Service
     */
    public static function fromJsonThing(JsonThing $jsonThing)
    {
        return new self(
            $jsonThing['Name'],
            $jsonThing['Action'],
//            $jsonThing['Version'],
            $jsonThing->hasProperty('Fields') ? $jsonThing['Fields']->map(
                function (JsonThing $jsonThing): Field {
                    return Field::fromJsonThing($jsonThing);
                }
            ) : []
        );
    }

    /**
     * @return array
     */
    public function toSerializable()
    {
        return $this->jsonSerialize();
    }

    /**
     * Specify data which should be serialized to JSON.
     *
     * @see http://php.net/manual/en/jsonserializable.jsonserialize.php
     *
     * @return mixed data which can be serialized by <b>json_encode</b>,
     * which is a value of any type other than a resource
     *
     * @since 5.4.0
     */
    public function jsonSerialize()
    {
        return [
            'Name' => $this->name,
            'Action' => $this->action,
            'Version' => $this->version,
            'Fields' => $this->getFields(),
        ];
    }
}
