<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Cli;

use IssetBV\PaymentBundle\CommandBus\SynchronizeAutoTopupProfiles\SynchronizeAutoTopupProfilesCommand;
use IssetBV\PaymentBundle\CommandBus\SynchronizeIssuers\SynchronizeIssuersCommand;
use IssetBV\PaymentBundle\CommandBus\SynchronizeWalletTypes\SynchronizeWalletTypesCommand;
use IssetBV\TalosBundle\Entity\TalosPaymentMethod;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\ResponseHandler\LoggingResponseHandler;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Logger\ConsoleLogger;
use Symfony\Component\Console\Output\NullOutput;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Debug\BufferingLogger;

/**
 * Class SynchronizeTypesCommand.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SynchronizeCommand extends Command
{
    /**
     * @var ApiGateway
     */
    private $apiGateway;

    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var LoggingResponseHandler
     */
    private $loggingResponseHandler;

    /**
     * @var OutputInterface
     */
    private $output;

    public function __construct(CommandBus $commandBus, ApiGateway $apiGateway, LoggingResponseHandler $loggingResponseHandler)
    {
        parent::__construct('talos:sync');

        $this->output = new NullOutput();
        $this->commandBus = $commandBus;
        $this->apiGateway = $apiGateway;
        $this->loggingResponseHandler = $loggingResponseHandler;
    }

    public function wrap(string $actionName, callable $action)
    {
        $this->output->write('> Synchronizing ' . $actionName . '... ');
        try {
            $action();
            $this->output->writeln('<info>OK</info>');
        } catch (\Exception $e) {
            $this->output->writeln('<error>ERR</error>');
        }
    }

    protected function configure()
    {
        $this->addOption('no-wallet', 'nw', InputOption::VALUE_NONE, 'Exclude wallet related synchronization commands');
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int|null|void
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->output = $output;

        $bufferingLogger = new BufferingLogger();
        //        $this->apiGateway->injectLogger($bufferingLogger);
        $this->loggingResponseHandler->injectLogger($bufferingLogger);

        // Load ideal issuers
        $this->wrap('issuers', function () {
            $this->commandBus->handle(new SynchronizeIssuersCommand(TalosPaymentMethod::SERVICE_NAME_IDEAL));
        });

        if (false === $input->getOption('no-wallet')) {
            $this->wrap('wallet types', function () {
                $this->commandBus->handle(new SynchronizeWalletTypesCommand());
            });
            $this->wrap('auto topup profiles', function () {
                $this->commandBus->handle(new SynchronizeAutoTopupProfilesCommand());
            });
        }

        $consoleLogger = new ConsoleLogger($output);
        \Functional\each($bufferingLogger->cleanLogs(), function ($log) use ($consoleLogger) {
            $consoleLogger->log(...$log);
        });
    }
}
