<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Service;

use Generator;
use InvalidArgumentException;
use IssetBV\PaymentBundle\CommandBus\ExecutePayment\CannotExecutePaymentException;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Payment\PaymentType;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Exception\ResponseException;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Response\Handler\ThrowingErrorHandler;
use IssetBV\TalosBundle\Gateway\Shared\GroupField;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\Factory\PaymentResponseHandlerFactory;
use Money\Formatter\DecimalMoneyFormatter;
use Money\MoneyFormatter;
use Symfony\Component\PropertyAccess\PropertyAccess;

/**
 * @author Tim Fennis <tim@isset.nl>
 *
 * @version 0.4.0
 */
class PaymentExecutor
{
    /**
     * @var MoneyFormatter
     */
    private $formatter;

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentResponseHandlerFactory
     */
    private $paymentResponseHandlerFactory;

    /**
     * PaymentExecutor constructor.
     *
     * @param DecimalMoneyFormatter $formatter
     * @param ApiGateway $gateway
     * @param PaymentResponseHandlerFactory $paymentResponseHandler
     */
    public function __construct(DecimalMoneyFormatter $formatter, ApiGateway $gateway, PaymentResponseHandlerFactory $paymentResponseHandler)
    {
        $this->gateway = $gateway;
        $this->formatter = $formatter;
        $this->paymentResponseHandlerFactory = $paymentResponseHandler;
    }

    public function executePayment(ExecutablePayment $payment, ClientData $clientData, string $returnUrl = '')
    {
        $service = $this->createServiceRequest($payment, iterator_to_array($this->createFields($payment)));

        $request = new Request($clientData, [$service], $returnUrl);
        $request->addResponseHandler($this->paymentResponseHandlerFactory->createResponseHandlerPayment($payment));
        $request->addRequestErrorHandler(new ThrowingErrorHandler());

        $channel = 'PayRecurrent' === $service->getAction()
            ? 'BatchFile'
            : 'Web';

        try {
            $this->gateway->send([$request], $channel);
        } catch (ResponseException $e) {
            throw CannotExecutePaymentException::becauseOfAResponseException($e);
        }
    }

    private function createServiceRequest(ExecutablePayment $payment, array $fields = [])
    {
        $serviceName = PropertyAccess::createPropertyAccessor()->getValue($payment->getPaymentMethod(), 'serviceName');

        $actionName = $payment->getOriginalPayment()->isDefined()
            ? 'PayRecurrent'
            : 'Pay';

        return new Service($serviceName, $actionName, $fields);
    }

    /**
     * @param ExecutablePayment $payment
     *
     * @return Generator
     */
    private function createFields(ExecutablePayment $payment)
    {
        yield $this->createPayment($payment);

        if ($payment->getPaymentIssuer()->isDefined()) {
            yield new SingleField('Issuer', $payment->getPaymentIssuer()->get()->getCode());
        }

        if ($payment->getOriginalPayment()->isDefined()) {
            /** @var Payment $payment */
            $payment = $payment->getOriginalPayment()->get();
            yield new SingleField('OriginalPaymentKey', $payment->getRemoteIdentifier()->get());
        }
    }

    /**
     * @param ExecutablePayment $payment
     *
     * @return GroupField
     */
    private function createPayment(ExecutablePayment $payment): GroupField
    {
        return new GroupField('Payment', [
            new SingleField('InvoiceNumber', $payment->getInvoice()->getNumber()->getAsString()),
            new SingleField('Reference', $payment->getReference()),
            new SingleField('Description', $payment->getDescription()->getOrElse('')),
            new SingleField('Currency', $payment->getAmount()->getCurrency()->getCode()),
            new SingleField('RecurrentType', $this->paymentTypeToString($payment->getType())),
            new SingleField('AmountDebit', $this->formatter->format($payment->getAmount())),
        ]);
    }

    /**
     * @param PaymentType $type
     *
     * @throws InvalidArgumentException
     *
     * @return string
     */
    private function paymentTypeToString(PaymentType $type): string
    {
        if ($type->isRecurring()) {
            return 'Recurring';
        }

        if ($type->isFirst()) {
            return 'First';
        }

        if ($type->isOneOff()) {
            return 'OneOff';
        }

        throw new InvalidArgumentException('Unknown payment type');
    }
}
