<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Payment\PaymentType;
use IssetBV\PaymentBundle\Domain\PaymentIssuer;
use IssetBV\PaymentBundle\Domain\Repository\PaymentIssuerRepository;
use IssetBV\PaymentBundle\Entity\Interval;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use IssetBV\TalosBundle\Storage\EntityStore;
use League\Tactician\CommandBus;
use Money\Money;
use Money\MoneyParser;
use Money\Parser\AggregateMoneyParser;
use PhpOption\Option;
use RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class CreateSubscriptionCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var PaymentIssuerRepository
     */
    private $issuerRepository;

    /**
     * @var MoneyParser
     */
    private $moneyParser;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var EntityStore
     */
    private $entityStore;

    public function __construct(
        CommandBus $commandBus,
        PaymentFactory $paymentFactory,
        InvoiceFactory $invoiceFactory,
        PaymentIssuerRepository $issuerRepository,
        AggregateMoneyParser $aggregateMoneyParser,
        EntityStore $entityStore
    ) {
        parent::__construct('subscription:create');
        $this->commandBus = $commandBus;
        $this->paymentFactory = $paymentFactory;
        $this->invoiceFactory = $invoiceFactory;
        $this->issuerRepository = $issuerRepository;
        $this->moneyParser = $aggregateMoneyParser;
        $this->entityStore = $entityStore;
    }

    protected function configure()
    {
        $this->addOption('payment-amount', null, InputOption::VALUE_REQUIRED, 'Create a payment for this subscription with an amount');
        $this->addOption('duration', null, InputOption::VALUE_REQUIRED, 'The duration of this subscription', '1 month');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $console = new SymfonyStyle($input, $output);

        $paymentIdOrNull = Option::fromValue($input->getOption('payment-amount'))
            ->flatMap(function ($payment) {
                return Option::fromValue($this->moneyParser->parse($payment, 'EUR'));
            })
            ->map(function (Money $subscriptionCost) {
                /** @var PaymentIssuer $paymentIssuer */
                $paymentIssuer = $this->issuerRepository
                    ->findOneByCode('INGBNL2A')
                    ->getOrThrow(new RuntimeException());

                return $this->paymentFactory->createExecutablePayment(
                    $this->invoiceFactory->createInvoice(),
                    $subscriptionCost,
                    $paymentIssuer->getPaymentMethod(),
                    $paymentIssuer,
                    PaymentType::createFirst()
                );
            })
            ->map(function (Payment $payment) {
                return $payment;
            })
            ->forAll(function (Payment $payment) use ($console) {
                $console->note('Payment created with ID: ' . $payment->getId()->asString());
            })
            ->getOrElse(null);

        $subscription = Subscription::createSubscription(
            Interval::createFromString($input->getOption('duration')),
            new \DateTime(),
            $paymentIdOrNull
        );

        $this->entityStore->persist($subscription);

        $console->success(sprintf('Subscription created with id %s', (string) $subscription->getId()));
    }
}
