<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Entity;

use DateInterval;
use DateTime;
use IssetBV\PaymentBundle\Domain\Exception\InvalidPaymentIntervalException;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\TalosBundle\Domain\TalosStatus;
use PhpOption\None;
use PhpOption\Option;
use PHPUnit\Framework\TestCase;
use Prophecy\Argument;

class SubscriptionTermTest extends TestCase
{
    public function test__construct()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);

        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));

        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertInstanceOf(SubscriptionTerm::class, $subscriptionTerm);
    }

    /**
     * @dataProvider hasSuccessfulPaymentDataProvider
     *
     * @param $paymentStatus
     * @param $expectedResponse
     */
    public function testHasSuccessfulPayment($paymentStatus, $expectedResponse)
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);
        $payment->getRemoteStatus()->shouldBeCalled()->willReturn($paymentStatus);

        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));

        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertInstanceOf(SubscriptionTerm::class, $subscriptionTerm);
        TestCase::assertEquals($expectedResponse, $subscriptionTerm->hasSuccessfulPayment());
    }

    public function testHasSuccessfulPaymentWithoutPayment()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            null,
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertInstanceOf(SubscriptionTerm::class, $subscriptionTerm);
        TestCase::assertEquals(false, $subscriptionTerm->hasSuccessfulPayment());
    }

    /**
     * @dataProvider isValidDataProvider
     *
     * @param $paymentStatus
     * @param bool $activityOverride
     * @param bool $expectedResponse
     */
    public function testIsValid($paymentStatus, bool $activityOverride, bool $expectedResponse)
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);
        $payment->getRemoteStatus()->shouldBeCalled()->willReturn($paymentStatus);

        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertInstanceOf(SubscriptionTerm::class, $subscriptionTerm);
        TestCase::assertEquals($expectedResponse, $subscriptionTerm->isValid());
    }

    public function testGetTermInterval()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);


        $intervalOption = $this->prophesize(Option::class);
        $intervalOption
            ->getOrThrow(Argument::type(InvalidPaymentIntervalException::class))
            ->shouldBeCalled()
            ->willReturn(new DateInterval('P1Y'));
        $intervalOption->get()->willReturn(new DateInterval('P1Y'));

        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn($intervalOption);

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            false
        );

        TestCase::assertInstanceOf(SubscriptionTerm::class, $subscriptionTerm);
        TestCase::assertEquals(new DateInterval('P1Y'), $subscriptionTerm->getTermInterval());
    }

    public function testGetDateTime()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertEquals($startTime, $subscriptionTerm->getDateTime());
    }

    public function testGetSubscription()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertEquals($subscription->reveal(), $subscriptionTerm->getSubscription());
    }

    public function testGetPayment()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertInstanceOf(Option::class, $subscriptionTerm->getPayment());
        TestCase::assertEquals($payment->reveal(), $subscriptionTerm->getPayment()->get());
    }

    public function testIsOverrideEnabled()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00');
        $payment = $this->prophesize(Payment::class);
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertEquals($activityOverride, $subscriptionTerm->isOverrideEnabled());
    }

    public function testIsActive()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('-1 week'); // today - 1 week
        $payment = $this->prophesize(Payment::class);
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y'))); // 1 year
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertEquals(true, $subscriptionTerm->isActive());

        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('-5 year'); // today - 1 week
        $payment = $this->prophesize(Payment::class);
        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y'))); // 1 year
        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertEquals(false, $subscriptionTerm->isActive());
    }

    public function testGetEndDate()
    {
        $subscription = $this->prophesize(Subscription::class);
        $startTime = new DateTime('2017-01-31T12:00:00+00:00');
        $payment = $this->prophesize(Payment::class);

        $termInterval = $this->prophesize(Interval::class);
        $termInterval->getDateInterval()->willReturn(Option::fromValue(new DateInterval('P1Y')));

        $activityOverride = false;

        $subscriptionTerm = new SubscriptionTerm(
            $subscription->reveal(),
            $startTime,
            $payment->reveal(),
            $termInterval->reveal(),
            $activityOverride
        );

        TestCase::assertEquals(new DateTime('2018-01-31T12:00:00+00:00'), $subscriptionTerm->getEndDate());

        // You are forced to refactor this test if you ever want to change the return type to Immutable! (which would be better)
        TestCase::assertInstanceOf(DateTime::class, $subscriptionTerm->getEndDate());

        // Check that the only change is 1 year!
        TestCase::assertEquals('2017-01-31T12:00:00+00:00', $startTime->format('c'));
        TestCase::assertEquals('2018-01-31T12:00:00+00:00', $subscriptionTerm->getEndDate()->format('c'));
    }

    public function hasSuccessfulPaymentDataProvider()
    {
        return [
            [TalosStatus::NONE, false],
            [TalosStatus::SUCCESS, true],
            [TalosStatus::PROCESSING, false],
            [TalosStatus::ON_HOLD, false],
            [TalosStatus::PENDING_APPROVAL, false],
            [TalosStatus::AWAITING_USER, false],
            [TalosStatus::FAILED, false],
            [TalosStatus::REJECTED, false],
            [TalosStatus::CANCELED_BY_USER, false],
            [TalosStatus::CANCELED_BY_BACK_OFFICE, false],
            [TalosStatus::TECHNICAL_ERROR, false],
            [TalosStatus::EXPIRED, false],
        ];
    }

    public function isValidDataProvider()
    {
        return [
            // If activity override is true always expect valid
            [TalosStatus::NONE, true, true],
            [TalosStatus::SUCCESS, true, true],
            [TalosStatus::PROCESSING, true, true],
            [TalosStatus::ON_HOLD, true, true],
            [TalosStatus::PENDING_APPROVAL, true, true],
            [TalosStatus::AWAITING_USER, true, true],
            [TalosStatus::FAILED, true, true],
            [TalosStatus::REJECTED, true, true],
            [TalosStatus::CANCELED_BY_USER, true, true],
            [TalosStatus::CANCELED_BY_BACK_OFFICE, true, true],
            [TalosStatus::TECHNICAL_ERROR, true, true],
            [TalosStatus::EXPIRED, true, true],

            // if activity override is false, expect true only if the payment is successful
            [TalosStatus::NONE, false, false],
            [TalosStatus::SUCCESS, false, true],
            [TalosStatus::PROCESSING, false, false],
            [TalosStatus::ON_HOLD, false, false],
            [TalosStatus::PENDING_APPROVAL, false, false],
            [TalosStatus::AWAITING_USER, false, false],
            [TalosStatus::FAILED, false, false],
            [TalosStatus::REJECTED, false, false],
            [TalosStatus::CANCELED_BY_USER, false, false],
            [TalosStatus::CANCELED_BY_BACK_OFFICE, false, false],
            [TalosStatus::TECHNICAL_ERROR, false, false],
            [TalosStatus::EXPIRED, false, false],
        ];
    }


}
