<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli\Renderer;

use DateTime;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\RenewAmountProvider;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use Money\MoneyFormatter;
use PhpOption\Option;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class SubscriptionRenderer
{
    /**
     * @var MoneyFormatter
     */
    private $formatter;

    /**
     * @var RenewAmountProvider
     */
    private $renewAmountProvider;

    public function __construct(MoneyFormatter $formatter, RenewAmountProvider $renewAmountProvider)
    {
        $this->formatter = $formatter;
        $this->renewAmountProvider = $renewAmountProvider;
    }

    public function render(Subscription $subscription, OutputInterface $output)
    {
        $table = new Table($output);
        $table->setHeaders(['Property', 'Value']);
        $table->addRows($this->getRows($subscription));
        $table->render();
    }

    public function renderWithStyle(Subscription $subscription, SymfonyStyle $style)
    {
        $style->table(['Property', 'Value'], $this->getRows($subscription));
    }

    private function getRows(Subscription $subscription)
    {
        return [
            ['id', $subscription->getId()],
            ['status', (string) $subscription->getStatus() . ' (' . $subscription->getStatus()->getCode() . ')'],
            ['start date', $subscription->getStartDate()->format('c')],
            ['date valid until', $this->date($subscription->getDateValidUntil())],
            ['date canceled', $this->date($subscription->getDateCanceled())],
            ['payment interval', $subscription->getPaymentInterval()->format('%y years + %m months + %d days + %h hours + %i minutes + %s seconds')],
            ['renew amount', $this->getRenewAmount($subscription)],
            ['original payment', $this->payment($subscription->getOriginalPayment())],
            ['renewable payment', $this->payment($subscription->findRenewablePayment())],
        ];
    }

    private function getRenewAmount(Subscription $subscription): string
    {
        try {
            return $this->formatter->format($this->renewAmountProvider->findRenewAmount($subscription));
        } catch (\RuntimeException $e) {
            return '-';
        }
    }

    /**
     * @param Option $dateCanceled
     *
     * @return string
     */
    private function date(Option $dateCanceled): string
    {
        return $dateCanceled
            ->map(function (DateTime $dateTime) {
                return $dateTime->format('c');
            })
            ->getOrElse('-');
    }

    private function payment(Option $payment)
    {
        return $payment
            ->map(function (Payment $payment) {
                return $payment->getRemoteIdentifier()->getOrElse('-') . ' / ' . $payment->getRemoteStatus() . ' / ' . $payment->getType();
            })
            ->getOrElse('-');
    }
}
