<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway\Request;

use Doctrine\Common\Collections\ArrayCollection;
use IssetBV\TalosBundle\Gateway\Response\Handler\RequestResultHandlerAware;
use JsonSerializable;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;

/**
 * Class RequestBase.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class Request implements JsonSerializable
{
    use RequestResultHandlerAware;

    /**
     * @var ClientData
     */
    private $clientData;

    /**
     * @var string
     */
    private $userCulture;

    /**
     * @var \DateTime
     */
    private $time;

    /**
     * @var UuidInterface
     */
    private $nonce;

    /**
     * this is super useful to have on EVERY request O.o.
     *
     * @var
     */
    private $returnUrl;

    /**
     * @var null
     */
    private $notifications;

    /**
     * @var string|null
     */
    private $notificationBehavior;

    /**
     * @var Service[]
     */
    private $services;

    /**
     * RequestBase constructor.
     *
     * @param ClientData $clientData
     * @param array $services
     * @param string $returnUrl
     */
    public function __construct(ClientData $clientData, array $services = [], $returnUrl = '')
    {
        $this->clientData = $clientData;
        $this->userCulture = 'nl-NL';
        $this->time = new \DateTime('now', new \DateTimeZone('UTC'));
        $this->nonce = Uuid::uuid4();
        $this->returnUrl = $returnUrl;
        $this->notifications = null;
        $this->notificationBehavior = null;
        $this->services = $services;
        $this->responseHandlers = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getNonce(): string
    {
        return (string) $this->nonce->getInteger();
    }

    /**
     * @deprecated use jsonSerialize instead
     *
     * @return array
     */
    public function toSerializable(): array
    {
        return $this->jsonSerialize();
    }

    public function jsonSerialize()
    {
        return [
            'ClientData' => $this->clientData->toSerializable(),
            'UserCulture' => $this->userCulture,
            'Time' => $this->time->format('c'),
            'Nonce' => (string) $this->nonce->getInteger(),
            'ReturnUrl' => $this->returnUrl,
            'Services' => array_map(
                function (Service $service) {
                    return $service->toSerializable();
                }, $this->services
            ),
            // ignore the other properties for the moment
        ];
    }
}
