<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\SynchronizeWallet;

use IssetBV\PaymentBundle\CommandBus\CreateWallet\CannotCreateWalletException;
use IssetBV\PaymentBundle\CommandBus\SynchronizeWallet\SynchronizeWalletCommand;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\MutationResponseHandler;
use IssetBV\TalosBundle\ResponseHandler\WalletResponseHandler;

/**
 * Class TalosSyncWalletHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SynchronizeWalletHandler
{
    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var MutationResponseHandler
     */
    private $mutationResponseHandler;

    /**
     * @var WalletResponseHandler
     */
    private $walletResponseHandler;

    /**
     * SynchronizeWalletHandler constructor.
     *
     * @param ApiGateway $gateway
     * @param \IssetBV\TalosBundle\ResponseHandler\MutationResponseHandler $mutationResponseHandler
     * @param \IssetBV\TalosBundle\ResponseHandler\WalletResponseHandler $walletResponseHandler
     */
    public function __construct(
        ApiGateway $gateway,
        MutationResponseHandler $mutationResponseHandler,
        WalletResponseHandler $walletResponseHandler
    ) {
        $this->gateway = $gateway;
        $this->mutationResponseHandler = $mutationResponseHandler;
        $this->walletResponseHandler = $walletResponseHandler;
    }

    /**
     * @param SynchronizeWalletCommand $command
     *
     * @throws CannotCreateWalletException
     */
    public function handle(SynchronizeWalletCommand $command)
    {
        $walletRequest = new Request(
            ClientData::empty(), [
                new Service('WalletSystem', 'Status', [
                    new SingleField('ObjectType', 'Wallet'),
                    new SingleField('ObjectIdentifier', $command->getWalletIdentifier()),
                ]),
            ]
        );

        $walletRequest->addResponseHandler($this->walletResponseHandler);

        $mutationRequest = new Request(ClientData::empty(), [
            new Service('WalletSystem', 'Status', [
                new SingleField('ObjectType', 'WalletMutation'),
                new SingleField('WalletKey', $command->getWalletIdentifier()),
            ]),
        ]);

        $mutationRequest->addResponseHandler($this->mutationResponseHandler);

        $this->gateway->send([$walletRequest, $mutationRequest]);
    }
}
