<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\ExecutePayment;

use IssetBV\PaymentBundle\CommandBus\ExecutePayment\CannotExecutePaymentException;
use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Entity\TalosPaymentMethod;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Service\PaymentExecutor;

/**
 * This class handles `ExecutePaymentCommand`s. By sending the necessary information to the payment gateway.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class ExecutePaymentHandler
{
    /**
     * @var PaymentExecutor
     */
    private $paymentExecutor;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * ExecutePaymentHandler constructor.
     *
     * @param PaymentRepository $paymentRepository
     * @param PaymentExecutor $paymentExecutor
     */
    public function __construct(PaymentRepository $paymentRepository, PaymentExecutor $paymentExecutor)
    {
        $this->paymentRepository = $paymentRepository;
        $this->paymentExecutor = $paymentExecutor;
    }

    /**
     * @param ExecutePaymentCommand $command
     *
     * @throws CannotExecutePaymentException
     */
    public function handle(ExecutePaymentCommand $command)
    {
        $paymentMethod = $this->findPaymentMethod($command);
        if (!$paymentMethod->isActive()) {
            throw CannotExecutePaymentException::becauseThePaymentMethodIsNotActive($paymentMethod);
        }

        $payment = $this->findPayment($command->getPaymentId());
        $clientData = $command->getClientData()->getOrElse(ClientData::empty());

        $this->paymentExecutor->executePayment($payment, $clientData);
    }

    private function findPayment(LocalPaymentIdentifier $paymentId): ExecutablePayment
    {
        return $this->paymentRepository
            ->optionallyFind($paymentId)
            ->getOrThrow(CannotExecutePaymentException::becauseThePaymentDoesNotExist($paymentId));
    }

    private function findPaymentMethod(ExecutePaymentCommand $command): TalosPaymentMethod
    {
        $paymentMethod = $this
            ->findPayment($command->getPaymentId())
            ->getPaymentMethod();

        if ($paymentMethod instanceof TalosPaymentMethod) {
            return $paymentMethod;
        }

        throw new CannotExecutePaymentException(self::class . ' cannot handle instances of ' . get_class($paymentMethod));
    }
}
