<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Domain\Subscription;

use Assert\Assertion;
use function Functional\map;
use function Functional\true;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class SubscriptionStatus
{
    const PENDING = 1;
    const VALID = 2;
    const EXPIRED = 3;
    const INVALIDATED = 5;

    const INVALID = 99;

    const VALID_STATUSES = [
        self::PENDING,
        self::VALID,
        self::EXPIRED,
        self::INVALIDATED,
        self::INVALID,
    ];

    private $status;

    public function __construct(int $status)
    {
        Assertion::inArray($status, self::VALID_STATUSES);
        $this->status = $status;
    }

    public function __toString()
    {
        switch ($this->status) {
            case self::PENDING:
                return 'pending';
            case self::VALID:
                return 'valid';
            case self::EXPIRED:
                return 'expired';
            case self::INVALIDATED:
                return 'invalidated';
            default:
                return 'unknown';
        }
    }

    /**
     * This constructor allows you to create a SubscriptionStatus value object for any type of integer value. If the
     * integer is outside of the allowed integer range then it's automatically converted to INVALID. This way we can
     * load denormalized statuses from the database without breaking the application with InvalidArgumentExceptions.
     *
     * @param int $denormalizedStatus
     *
     * @return SubscriptionStatus
     */
    public static function safelyFromInteger(int $denormalizedStatus): SubscriptionStatus
    {
        try {
            return new self($denormalizedStatus);
        } catch (\InvalidArgumentException $invalidArgumentException) {
            return new self(self::INVALID);
        }
    }

    /**
     * @param string $subscriptionStatus
     *
     * @return SubscriptionStatus
     */
    public static function fromString(string $subscriptionStatus): SubscriptionStatus
    {
        switch ($subscriptionStatus) {
            case 'pending':
                return new self(self::PENDING);
            case 'valid':
                return new self(self::VALID);
            case 'expired':
                return new self(self::EXPIRED);
            case 'invalidated':
                return new self(self::INVALIDATED);
        }

        throw new \InvalidArgumentException($subscriptionStatus . ' is no (longer a) valid status');
    }

    public function getCode(): int
    {
        return $this->status;
    }

    public static function invalidated()
    {
        return new self(self::INVALIDATED);
    }

    public static function valid()
    {
        return new self(self::VALID);
    }

    public static function pending()
    {
        return new self(self::PENDING);
    }

    public static function expired()
    {
        return new self(self::EXPIRED);
    }

    public function isPending()
    {
        return $this->status === self::PENDING;
    }

    public function isInvalidated()
    {
        return $this->status === self::INVALIDATED;
    }

    public function isValid()
    {
        return $this->status === self::VALID;
    }

    public function isExpired()
    {
        return $this->status === self::EXPIRED;
    }

    public function equals($object)
    {
        if ($object instanceof self) {
            return $object->getCode() === $this->getCode();
        }

        return false;
    }

    /**
     * @param callable[] $functions
     */
    public function handleAll($functions)
    {
        $valid = true(map(self::VALID_STATUSES, function ($status) use ($functions) {
            return array_key_exists($status, $functions);
        }));
        if (false === $valid) {
            throw new \LogicException('You are not handling one of the possible values of SubscriptionStatus');
        }

        $functions[$this->getCode()]();
    }
}
